<?php
// notbank_historial_withdraw.php
//
// Este script amplía la funcionalidad de consulta de transacciones para permitir
// seleccionar una transacción y crear un retiro de criptomonedas a una
// dirección especificada. Utiliza la API de Notbank para autenticarse,
// listar transacciones y enviar un retiro a la red blockchain a través del
// endpoint CreateCryptoWithdraw. Las direcciones de destino deben estar
// previamente whitelisteadas según la documentación de Notbank【261154067702217†screenshot】.

/*
 * ADVERTENCIA: El envío de fondos es una operación sensible. Este script es
 * solo un ejemplo educativo que demuestra cómo estructurar las llamadas a
 * la API. No ejecuta ninguna transacción automáticamente y asume que ya
 * existen direcciones whitelisteadas en la cuenta de origen. Pruebe en
 * entornos seguros y revise la documentación de Notbank antes de usarlo
 * en producción【261154067702217†screenshot】.
 */

// ----- Configuración -----
$apiBaseUrl  = 'https://api.notbank.exchange';
$apiKey      = '40b266e4f2f010631ddfb26cda4d915a';    // Llave pública (APIKey)
$apiSecret   = '5bc92a8df76d2d88248590e0fef53ecc';    // Llave privada (secreta)
$userId      = 1081;                                   // ID de usuario
$accountId   = 4231;                                   // ID de cuenta. Sustituye con tu account_id real para filtrar mejor

/**
 * Genera la firma HMAC-SHA256 necesaria para la autenticación.
 */
function nb_generateSignature($nonce, $userId, $apiKey, $apiSecret) {
    $payload = $nonce . $userId . $apiKey;
    return hash_hmac('sha256', $payload, $apiSecret);
}

/**
 * Realiza la autenticación con Notbank y devuelve el SessionToken (aptoken).
 */
function nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId) {
    // Autenticación basada en nonce de milisegundos y cabecera Content-Type
    // Usamos un nonce de alta resolución (ms) como recomienda Notbank para evitar duplicidades
    $nonce = (string) round(microtime(true) * 1000);
    // La firma se genera concatenando nonce + userId + apiKey y aplicando HMAC-SHA256 con la clave secreta
    $signature = nb_generateSignature($nonce, $userId, $apiKey, $apiSecret);
    $url = rtrim($apiBaseUrl, '/') . '/AP/AuthenticateUser';
    $headers = [
        'APIKey: ' . $apiKey,
        'Signature: ' . $signature,
        'UserId: ' . $userId,
        'Nonce: ' . $nonce,
        // Incluir Content-Type como en el ejemplo original de autenticación
        'Content-Type: application/json'
    ];
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER     => $headers
    ]);
    $response = curl_exec($ch);
    if (curl_errno($ch)) {
        curl_close($ch);
        return false;
    }
    curl_close($ch);
    $data = json_decode($response, true);
    // Notbank puede devolver SessionToken en el nivel superior o dentro de 'data'
    if (is_array($data)) {
        if (isset($data['SessionToken']) && !empty($data['SessionToken'])) {
            return $data['SessionToken'];
        }
        if (isset($data['data']['SessionToken']) && !empty($data['data']['SessionToken'])) {
            return $data['data']['SessionToken'];
        }
        if (isset($data['sessionToken']) && !empty($data['sessionToken'])) {
            return $data['sessionToken'];
        }
    }
    return false;
}

/**
 * Obtiene todas las transacciones de la cuenta especificada.
 *
 * @param string $apiBaseUrl  Base URL de la API
 * @param string $aptoken     Token de sesión
 * @param int    $accountId   ID de la cuenta
 * @param string|null $fromDate Fecha inicial (dd/mm/yyyy) opcional
 * @param string|null $toDate   Fecha final (dd/mm/yyyy) opcional
 * @return array               Listado de transacciones
 */
function nb_getAllTransactions($apiBaseUrl, $aptoken, $accountId, $fromDate = null, $toDate = null) {
    $page = 1;
    $pageSize = 50;
    $allData = [];
    do {
        $params = [
            'page'      => $page,
            'page_size' => $pageSize
        ];
        if ($accountId) {
            $params['accountId'] = $accountId;
        }
        if ($fromDate) {
            $params['from_date'] = $fromDate;
        }
        if ($toDate) {
            $params['to_date'] = $toDate;
        }
        $url = rtrim($apiBaseUrl, '/') . '/api/nb/wallet/transactions?' . http_build_query($params);
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER     => [
                'Content-Type: application/json',
                'aptoken: ' . $aptoken
            ]
        ]);
        $response = curl_exec($ch);
        if (curl_errno($ch)) {
            curl_close($ch);
            break;
        }
        curl_close($ch);
        $result = json_decode($response, true);
        $data = $result['data'] ?? [];
        $allData = array_merge($allData, $data);
        $page++;
        $hasMore = count($data) >= $pageSize;
    } while ($hasMore);
    return $allData;
}

/**
 * Envía una solicitud de retiro de criptomonedas (CreateCryptoWithdraw).
 *
 * @param string $apiBaseUrl   URL base de Notbank
 * @param string $aptoken      Token de sesión
 * @param int    $accountId    ID de la cuenta origen
 * @param string $currency     Código de la moneda (p.ej. "USDT", "BTC")
 * @param string $network      Red de la moneda (p.ej. "USDT_BSC")
 * @param string $address      Dirección destino
 * @param float  $amount       Monto a retirar
 * @param string|null $memoTag Memo o tag (si la red lo requiere)
 * @param string|null $otp     Código de autenticación en dos pasos
 * @return array|false         Respuesta de la API decodificada o false en caso de error
 */
function nb_createCryptoWithdraw($apiBaseUrl, $aptoken, $accountId, $currency, $network, $address, $amount, $memoTag = null, $otp = null) {
    $url = rtrim($apiBaseUrl, '/') . '/api/nb/wallet/crypto/withdrawal';
    $payload = [
        'account_id' => (int)$accountId,
        'currency'   => $currency,
        'network'    => $network,
        'address'    => $address,
        'amount'     => $amount
    ];
    if (!empty($memoTag)) {
        $payload['memo_or_tag'] = $memoTag;
    }
    if (!empty($otp)) {
        $payload['otp'] = $otp;
    }
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/json',
            'aptoken: ' . $aptoken
        ],
        CURLOPT_POSTFIELDS     => json_encode($payload)
    ]);
    $response = curl_exec($ch);
    if (curl_errno($ch)) {
        curl_close($ch);
        return false;
    }
    curl_close($ch);
    return json_decode($response, true);
}

// ------ Lógica del script ------
$fromDate      = isset($_POST['fromDate']) ? trim($_POST['fromDate']) : null;
$toDate        = isset($_POST['toDate'])   ? trim($_POST['toDate'])   : null;
$message       = null;
$transactions  = [];

// Verificar si se envió una solicitud de retiro
if (isset($_POST['action']) && $_POST['action'] === 'withdraw') {
    // Recoger datos del formulario de retiro
    $withdrawAccountId = (int)($_POST['source_account_id'] ?? 0);
    $currency   = $_POST['currency'] ?? '';
    $amount     = floatval($_POST['amount'] ?? 0);
    $network    = $_POST['network'] ?? '';
    $address    = $_POST['address'] ?? '';
    $memoTag    = $_POST['memo'] ?? null;
    $otp        = $_POST['otp'] ?? null;

    if ($withdrawAccountId && $currency && $amount > 0 && $network && $address) {
        $aptoken = nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId);
        if ($aptoken) {
            $response = nb_createCryptoWithdraw($apiBaseUrl, $aptoken, $withdrawAccountId, $currency, $network, $address, $amount, $memoTag, $otp);
            if ($response && isset($response['status']) && strtolower($response['status']) === 'success') {
                $message = 'Retiro solicitado con éxito. ID de solicitud: ' . htmlspecialchars($response['data']);
            } else {
                $message = 'Error al realizar el retiro: ' . htmlspecialchars(json_encode($response));
            }
        } else {
            $message = 'Error de autenticación al obtener el aptoken.';
        }
    } else {
        $message = 'Por favor complete todos los campos obligatorios para el retiro.';
    }
}

// Independientemente de si hubo retiro, obtener transacciones para mostrar
{
    $aptoken = nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId);
    if ($aptoken) {
        $transactions = nb_getAllTransactions($apiBaseUrl, $aptoken, $accountId, $fromDate ?: null, $toDate ?: null);
    } else {
        $message = 'No se pudo autenticar para obtener las transacciones.';
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Historial y retiros de criptomonedas - Notbank</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        form { margin-bottom: 10px; }
        table { border-collapse: collapse; width: 100%; margin-top: 20px; }
        th, td { border: 1px solid #ccc; padding: 8px; text-align: left; }
        th { background-color: #f4f4f4; }
        input[type="text"], input[type="number"], select { padding: 4px; width: 100%; }
        .withdraw-form { background: #f5f5f5; padding: 8px; border: 1px solid #ddd; }
        .message { margin-top: 20px; padding: 10px; border-radius: 5px; }
        .success { background-color: #e0f7e0; color: #0c6b0c; }
        .error { background-color: #fce4e4; color: #a94442; }
    </style>
</head>
<body>
    <h1>Historial de transacciones y retiros</h1>
    <p>Consulta el historial de movimientos de la cuenta y solicita retiros de criptomonedas basados en las transacciones existentes. Solo podrás enviar a direcciones whitelisteadas【261154067702217†screenshot】.</p>

    <form method="post" style="background:#fafafa; padding:10px; max-width:400px;">
        <label>Fecha desde (dd/mm/yyyy):
            <input type="text" name="fromDate" value="<?php echo htmlspecialchars($fromDate ?? ''); ?>" placeholder="Opcional">
        </label>
        <label>Fecha hasta (dd/mm/yyyy):
            <input type="text" name="toDate" value="<?php echo htmlspecialchars($toDate ?? ''); ?>" placeholder="Opcional">
        </label>
        <button type="submit">Filtrar historial</button>
    </form>

    <?php if ($message): ?>
        <div class="message <?php echo (strpos($message, 'éxito') !== false ? 'success' : 'error'); ?>">
            <?php echo $message; ?>
        </div>
    <?php endif; ?>

    <?php if (!empty($transactions)): ?>
    <table>
        <tr>
            <th>Fecha</th>
            <th>Moneda</th>
            <th>Dirección</th>
            <th>Monto</th>
            <th>Tipo</th>
            <th>Hash</th>
            <th>Retirar</th>
        </tr>
        <?php foreach ($transactions as $index => $tx): ?>
            <tr>
                <td><?php echo htmlspecialchars($tx['created_at'] ?? ''); ?></td>
                <td><?php echo htmlspecialchars($tx['currency'] ?? ''); ?></td>
                <td><?php echo htmlspecialchars($tx['address'] ?? ''); ?></td>
                <td><?php echo htmlspecialchars($tx['amount'] ?? ''); ?></td>
                <td><?php echo ((isset($tx['direction']) && $tx['direction'] == 1) ? 'Entrada' : 'Salida'); ?></td>
                <td><?php echo htmlspecialchars($tx['hash'] ?? ''); ?></td>
                <td>
                    <?php
                        // Prellenar datos para la transacción de retiro solo si es entrada (dirección externa)
                        $txAmount   = $tx['amount'] ?? '';
                        $txCurrency = $tx['currency'] ?? '';
                        // Determinar red predeterminada: si la moneda contiene '_', dividirla (p.ej. USDTBSC => USDT_BSC)
                        $defaultNetwork = '';
                        if (!empty($txCurrency)) {
                            // Convertir nombres del tipo "USDTBSC" a "USDT_BSC" por convención
                            $pattern = '/([A-Za-z]+)(BSC|ETH|TRC|SOL)/';
                            if (preg_match($pattern, $txCurrency, $matches)) {
                                $defaultNetwork = $matches[1] . '_' . $matches[2];
                                $txCurrency = $matches[1];
                            }
                        }
                    ?>
                    <form method="post" class="withdraw-form">
                        <input type="hidden" name="action" value="withdraw">
                        <input type="hidden" name="source_account_id" value="<?php echo (int)($tx['account_id'] ?? $accountId); ?>">
                        <input type="hidden" name="currency" value="<?php echo htmlspecialchars($txCurrency); ?>">
                        <input type="hidden" name="amount" value="<?php echo htmlspecialchars($txAmount); ?>">
                        <label>Red:<br>
                            <input type="text" name="network" value="<?php echo htmlspecialchars($defaultNetwork); ?>" required>
                        </label><br>
                        <label>Dirección destino:<br>
                            <input type="text" name="address" placeholder="0x..." required>
                        </label><br>
                        <label>Memo/Tag (opcional):<br>
                            <input type="text" name="memo" placeholder="">
                        </label><br>
                        <label>OTP (opcional):<br>
                            <input type="text" name="otp" placeholder="">
                        </label><br>
                        <button type="submit">Enviar</button>
                    </form>
                </td>
            </tr>
        <?php endforeach; ?>
    </table>
    <?php else: ?>
        <p>No se encontraron transacciones para los filtros seleccionados.</p>
    <?php endif; ?>
</body>
</html>