<?php
/**
 * rescate_async_v2_8_1.php
 * Rescate USDT_BSC · Depósito huérfano → Whitelist → Retiro (async, variantes A/B/C con idempotencia)
 * - API interna: ?api=1&op=auth|hist|deposit|wl|withdraw
 * - Frontend async con fetch()
 */

session_start();
ini_set('display_errors', 0);
error_reporting(E_ALL);

/* =======================================================
 * ===============  [1] CONFIGURACIÓN  ====================
 * ======================================================= */
$API_BASE   = 'https://api.notbank.exchange';
$API_KEY    = '40b266e4f2f010631ddfb26cda4d915a';
$API_SECRET = '5bc92a8df76d2d88248590e0fef53ecc';
$USER_ID    = 1081;
$ACCOUNT_ID = 4231;

$DEFAULT_CURRENCY = 'USDT';
$DEFAULT_NETWORK  = 'USDT_BSC';
$DEST_ADDRESS     = '0xc6c93fe38d5be2fe8065e6bdc2a4f57f8a45f471';


/* =======================================================
 * ===============  [2] HTTP CORE UTILS  =================
 * ======================================================= */
function parse_headers_lower($raw){
    $out = [];
    foreach (preg_split("/\r\n|\n|\r/", (string)$raw) as $line) {
        if (strpos($line, ':') !== false) {
            [$k,$v] = explode(':', $line, 2);
            $out[strtolower(trim($k))] = trim($v);
        }
    }
    return $out;
}

/** Cliente HTTP JSON (GET/POST), tolerante a BOM y HTML de error */
function http_json($method, $url, array $headers=[], $payload=null){
    $ch = curl_init($url);
    $h = $headers;
    $h[] = 'Accept: application/json';
    $h[] = 'Expect:';               // evita 100-continue
    $h[] = 'Connection: close';
    curl_setopt($ch, CURLOPT_ENCODING, ''); // gzip/deflate

    if (strtoupper($method)==='POST'){
        curl_setopt($ch, CURLOPT_POST, true);
        if ($payload===null){
            // Importante para Cloudflare 411
            curl_setopt($ch, CURLOPT_POSTFIELDS, '');
            $h[] = 'Content-Length: 0';
        } else {
            $body = is_string($payload) ? $payload : json_encode($payload);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
            $h[] = 'Content-Type: application/json';
        }
    } else {
        curl_setopt($ch, CURLOPT_HTTPGET, true);
    }

    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => 30,
        CURLOPT_HTTPHEADER     => $h,
        CURLOPT_HEADER         => true,
        CURLOPT_FAILONERROR    => false,
    ]);

    $resp = curl_exec($ch);
    $info = curl_getinfo($ch);
    curl_close($ch);

    $hs = $info['header_size'] ?? 0;
    $rawHeaders = substr($resp, 0, $hs);
    $rawBody    = substr($resp, $hs);

    $trimmed = ltrim($rawBody, "\xEF\xBB\xBF \t\r\n");
    $json = json_decode($trimmed, true);
    if (!is_array($json) && preg_match('/\{.*\}/s', $trimmed, $m)){
        $json = json_decode($m[0], true);
    }

    return [
        'http'     => (int)($info['http_code'] ?? 0),
        'body_raw' => $rawBody,
        'json'     => is_array($json) ? $json : null,
    ];
}

/* =======================================================
 * ==================  [3] HELPERS  ======================
 * ======================================================= */
function sign_auth($nonce, $userId, $apiKey, $apiSecret){
    $payload = (string)$nonce . (string)$userId . (string)$apiKey;
    return hash_hmac('sha256', $payload, $apiSecret);
}
function decstr($n, $d=8){
    $s = number_format((float)$n, $d, '.', '');
    $s = rtrim(rtrim($s, '0'), '.');
    return $s === '' ? '0' : $s;
}
function normalize_network($currency, $network){
    $c = strtoupper(trim($currency));
    $n = strtoupper(trim($network));
    $n = str_replace('-', '_', $n);
    $map = ['BEP20'=>'BSC','ERC20'=>'ETH','TRC20'=>'TRC'];
    if (isset($map[$n])) $n = $map[$n];
    if (preg_match('/^'.$c.'_(BSC|ETH|TRC|SOL)$/', $n)) return $n;
    if (in_array($n, ['BSC','ETH','TRC','SOL'], true)) return $c.'_'.$n;
    if (preg_match('/^([A-Z]+)(BSC|ETH|TRC|SOL)$/', $n, $m)) return $m[1].'_'.$m[2];
    return $n;
}
function uuidv4(){
    $d = random_bytes(16);
    $d[6] = chr((ord($d[6]) & 0x0f) | 0x40);
    $d[8] = chr((ord($d[8]) & 0x3f) | 0x80);
    return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($d), 4));
}

/* =======================================================
 * === [4] FUNCIÓN CLAVE: RETIRO CON VARIANTES A/B/C =====
 * ======================================================= */
/**
 * Intenta retiro probando variantes A→B→C con idempotencia.
 * - A: address + whitelist_id + (provider_id si hay)
 * - B: address + whitelist_id (sin provider_id)
 * - C: solo address
 * Maneja 401 required_token, 500 internal_error (pasa a siguiente).
 */
function nb_attemptWithdrawVariants($apiBaseUrl, $aptoken, $accountId, $currency, $network, $amountStr, $address, $whitelistId = null, $providerId = null, $otp = null) {
    $url = rtrim($apiBaseUrl, '/') . '/api/nb/wallet/crypto/withdrawal';

    $base = [
        'account_id' => (int)$accountId,
        'currency'   => strtoupper(trim($currency)),
        'network'    => normalize_network($currency, $network),
        'amount'     => $amountStr, // SIEMPRE string
    ];

    // A
    $va = $base + ['address' => $address];
    if (!empty($whitelistId)) $va['whitelist_id'] = $whitelistId;
    if (!empty($providerId))  $va['provider_id']  = (int)$providerId;
    if (!empty($otp))         $va['otp']          = $otp;

    // B (sin provider_id)
    $vb = $base + ['address' => $address];
    if (!empty($whitelistId)) $vb['whitelist_id'] = $whitelistId;
    if (!empty($otp))         $vb['otp']          = $otp;

    // C (solo address)
    $vc = $base + ['address' => $address];
    if (!empty($otp)) $vc['otp'] = $otp;

    $variants = [
        ['name' => 'A', 'data' => $va],
        ['name' => 'B', 'data' => $vb],
        ['name' => 'C', 'data' => $vc],
    ];

    $last = null;
    foreach ($variants as $v) {
        $headers = [
            'Content-Type: application/json',
            'Accept: application/json',
            'aptoken: ' . $aptoken,
            'X-Idempotency-Key: ' . uuidv4(), // idempotencia por variante
        ];
        $res = http_json('POST', $url, $headers, $v['data']);

        $http = $res['http'] ?? 0;
        $json = $res['json'] ?? null;

        if ($http === 401 && is_array($json) && ($json['code'] ?? '') === 'required_token') {
            return [
                'ok'        => false,
                'need_otp'  => true,
                'variant'   => $v['name'],
                'last_http' => $http,
                'last_body' => $res['body_raw'] ?? null,
                'payload'   => $v['data'],
            ];
        }

        if ($http >= 200 && $http < 300 && is_array($json) && strtolower($json['status'] ?? '') === 'success') {
            $id = $json['data']['id'] ?? ($json['data'] ?? null);
            return [
                'ok'      => true,
                'id'      => $id,
                'variant' => $v['name'],
            ];
        }

        if ($http !== 500) {
            return [
                'ok'        => false,
                'need_otp'  => false,
                'msg'       => strtolower($json['code'] ?? 'error'),
                'variant'   => $v['name'],
                'last_http' => $http,
                'last_body' => $res['body_raw'] ?? null,
                'payload'   => $v['data'],
            ];
        }

        // 500 → intenta la siguiente
        $last = $res;
        usleep(300000);
    }

    return [
        'ok'        => false,
        'need_otp'  => false,
        'msg'       => 'internal_error',
        'variant'   => 'A→C',
        'last_http' => $last['http'] ?? 0,
        'last_body' => $last['body_raw'] ?? null,
        'payload'   => $variants[0]['data'],
    ];
}

/* =======================================================
 * ===============  [5] OPERACIONES API  =================
 * ======================================================= */
function op_auth(){
    global $API_BASE, $API_KEY, $API_SECRET, $USER_ID;

    // antirebote 2s
    $now = microtime(true);
    if (!empty($_SESSION['aptoken']) && ($now - ($_SESSION['last_auth_ts'] ?? 0)) < 2.0){
        return ['ok'=>true, 'token'=>$_SESSION['aptoken'], 'http'=>200, 'body'=>'<cached>'];
    }

    // Nonce monótono ms
    $genNonce = function(){
        $ms = (int)round(microtime(true)*1000);
        $last = $_SESSION['last_nonce'] ?? 0;
        if ($ms <= $last) $ms = $last + 1;
        $_SESSION['last_nonce'] = $ms;
        return (string)$ms;
    };

    $attempts = 0; $last = ['http'=>0,'body_raw'=>'<no-body>'];
    while ($attempts++ < 3){
        $nonce = $genNonce();
        $sig   = sign_auth($nonce, $USER_ID, $API_KEY, $API_SECRET);

        $res = http_json('POST', rtrim($API_BASE,'/').'/AP/AuthenticateUser', [
            'APIKey: '    . $API_KEY,
            'Signature: ' . $sig,
            'UserId: '    . $USER_ID,
            'Nonce: '     . $nonce,
        ], null);

        $tok = null;
        if (is_array($res['json'])){
            $j = $res['json'];
            $tok = $j['SessionToken'] ?? $j['sessionToken'] ?? ($j['data']['SessionToken'] ?? null);
        }
        if (!$tok && !empty($res['body_raw'])){
            if (preg_match('/"SessionToken"\s*:\s*"([^"]+)"/i', $res['body_raw'], $m)) $tok = $m[1];
            elseif (preg_match('/"sessionToken"\s*:\s*"([^"]+)"/i', $res['body_raw'], $m)) $tok = $m[1];
        }

        if ($tok){
            $_SESSION['aptoken'] = $tok;
            $_SESSION['last_auth_ts'] = microtime(true);
            return ['ok'=>true, 'token'=>$tok, 'http'=>$res['http'], 'body'=>'<parsed>'];
        }

        usleep(250000);
        $last = $res;
    }

    return ['ok'=>false, 'token'=>null, 'http'=>$last['http'] ?? 0, 'body'=>$last['body_raw'] ?? '<no-body>'];
}

function auth_header(){
    $tok = $_SESSION['aptoken'] ?? '';
    return [
        "aptoken: $tok",
        "Content-Type: application/json"
    ];
}

function op_hist(){
    global $API_BASE;
    if (empty($_SESSION['aptoken'])) return ['ok'=>false,'rows'=>[],'raw'=>['http'=>0,'body_raw'=>'<no aptoken>']];
    $u = rtrim($API_BASE,'/').'/api/nb/wallet/transactions';
    $res = http_json('GET', $u, auth_header());
    if (in_array($res['http'], [401,403])) {
        $a = op_auth();
        if (!empty($a['token'])) {
            $res = http_json('GET', $u, auth_header());
        }
    }
    $rows = [];
    if ($res['http']===200 && is_array($res['json']['data'] ?? null)) {
        $rows = $res['json']['data'];
        usort($rows,function($x,$y){
            $A = strtotime($x['created_at'] ?? '') ?: 0;
            $B = strtotime($y['created_at'] ?? '') ?: 0;
            return $B <=> $A;
        });
        $rows = array_slice($rows, 0, 5);
    }
    return ['ok'=>$res['http']===200, 'rows'=>$rows, 'raw'=>$res];
}

function op_deposit_addr($accountId, $currency, $network){
    global $API_BASE;
    if (empty($_SESSION['aptoken'])) return ['ok'=>false,'address'=>'','raw'=>['http'=>0,'body_raw'=>'<no aptoken>']];
    $url = rtrim($API_BASE,'/').'/api/nb/wallet/crypto'
        .'?account_id='.(int)$accountId
        .'&currency='.urlencode(strtoupper($currency))
        .'&network='.urlencode(normalize_network($currency,$network));
    $res = http_json('GET', $url, auth_header());
    if (in_array($res['http'], [401,403])) { $a=op_auth(); if (!empty($a['token'])) $res = http_json('GET', $url, auth_header()); }
    $addr = '';
    if ($res['http']===200 && is_array($res['json']['data'] ?? null) && count($res['json']['data'])>0) {
        $addr = $res['json']['data'][0];
    }
    return ['ok'=>$res['http']===200, 'address'=>$addr, 'raw'=>$res];
}

function op_whitelist_list($accountId, $search){
    global $API_BASE;
    if (empty($_SESSION['aptoken'])) return ['ok'=>false,'verified'=>false,'whitelist_id'=>null,'provider_id'=>null,'raw'=>['http'=>0,'body_raw'=>'<no aptoken>']];
    $url = rtrim($API_BASE,'/').'/api/nb/wallet/crypto/whitelist-addresses'
        .'?account_id='.(int)$accountId.'&search='.urlencode($search);
    $res = http_json('GET', $url, auth_header());
    if (in_array($res['http'], [401,403])) { $a=op_auth(); if (!empty($a['token'])) $res = http_json('GET', $url, auth_header()); }

    $wl_id = null; $provider_id = null; $verified = false;
    if ($res['http']===200 && is_array($res['json']['data'] ?? null)) {
        foreach ($res['json']['data'] as $row) {
            if (strcasecmp($row['address']??'', $search)===0 && !empty($row['verified'])) {
                $wl_id = $row['id'] ?? null;
                $provider_id = $row['provider_id'] ?? null;
                $verified = true;
                break;
            }
        }
    }
    return ['ok'=>$res['http']===200, 'verified'=>$verified, 'whitelist_id'=>$wl_id, 'provider_id'=>$provider_id, 'raw'=>$res];
}

/** ======  NUEVO Paso 6: usa nb_attemptWithdrawVariants()  ====== */
function op_withdraw($accountId, $currency, $network, $destAddr, $amount, $variant, $otp=null){
    global $API_BASE;
    if (empty($_SESSION['aptoken'])) return ['ok'=>false,'msg'=>'no_aptoken','last'=>['raw'=>['http'=>0,'body_raw'=>'<no aptoken>'],'variant'=>$variant,'payload'=>[]]];

    // Prechequeo whitelist para traer wl_id y provider_id conocidos
    $pre = op_whitelist_list($accountId, $destAddr);
    $wlId = $pre['whitelist_id'] ?? null;
    $prov = $pre['provider_id'] ?? null;

    // Monto string
    $amountStr = decstr($amount);

    // Si el usuario elige explícitamente B o C, forzamos esa primera,
    // pero la función probará variantes siguientes si cae en 500.
    $order = ['A','B','C'];
    if (in_array($variant, ['B','C'], true)) {
        $order = array_values(array_unique(array_merge([$variant], $order)));
    }

    $last = null;
    foreach ($order as $firstVariant) {
        $res = nb_attemptWithdrawVariants(
            $API_BASE,
            $_SESSION['aptoken'],
            $accountId,
            $currency,
            $network,
            $amountStr,
            $destAddr,
            $wlId ?: null,
            ($firstVariant==='A' ? ($prov ?: null) : null), // provider solo en A
            $otp ?: null
        );

        if (!empty($res['ok'])) {
            return ['ok'=>true, 'id'=>$res['id'] ?? null, 'last'=>['variant'=>$res['variant'],'raw'=>['http'=>200,'body_raw'=>'{"status":"success"}'],'payload'=>[]]];
        }
        if (!empty($res['need_otp'])) {
            return ['ok'=>false,'need_otp'=>true,'msg'=>'required_token','last'=>['variant'=>$res['variant'],'raw'=>['http'=>$res['last_http'] ?? 401,'body_raw'=>$res['last_body'] ?? ''],'payload'=>$res['payload'] ?? []]];
        }

        // Si no fue 500, salir con el error devuelto
        if (($res['msg'] ?? '') !== 'internal_error') {
            return ['ok'=>false,'msg'=>($res['msg'] ?? 'error'),'last'=>['variant'=>$res['variant'],'raw'=>['http'=>$res['last_http'] ?? 0,'body_raw'=>$res['last_body'] ?? ''],'payload'=>$res['payload'] ?? []]];
        }

        // internal_error → probamos siguiente variante del orden local
        $last = $res;
        usleep(200000);
    }

    // Si todas terminaron en internal_error
    return ['ok'=>false,'msg'=>'internal_error','last'=>['variant'=>'A→C','raw'=>['http'=>$last['last_http'] ?? 500,'body_raw'=>$last['last_body'] ?? ''],'payload'=>$last['payload'] ?? []]];
}

/* =======================================================
 * ===============  [6] MINI API (AJAX)  =================
 * ======================================================= */
if (isset($_GET['api'])) {
    header('Content-Type: application/json; charset=utf-8');

    $op = $_GET['op'] ?? '';
    try {
        if ($op === 'auth') {
            $r = op_auth();
            echo json_encode([
                'ok'   => $r['ok'],
                'token'=> $r['token'] ? (substr($r['token'],0,8).'•••'.substr($r['token'],-4)) : null,
                'http' => $r['http'],
                'body' => $r['body'],
            ], JSON_UNESCAPED_UNICODE);
            exit;
        }
        if ($op === 'hist') {
            $r = op_hist();
            echo json_encode([
                'ok'       => $r['ok'],
                'http'     => $r['raw']['http'] ?? 0,
                'body'     => $r['raw']['body_raw'] ?? '',
                'rows'     => $r['rows'],
                'has_token'=> !empty($_SESSION['aptoken']),
            ], JSON_UNESCAPED_UNICODE);
            exit;
        }
        if ($op === 'deposit') {
            $r = op_deposit_addr($ACCOUNT_ID, $DEFAULT_CURRENCY, $DEFAULT_NETWORK);
            echo json_encode([
                'ok'       => $r['ok'],
                'http'     => $r['raw']['http'] ?? 0,
                'body'     => $r['raw']['body_raw'] ?? '',
                'address'  => $r['address'],
                'has_token'=> !empty($_SESSION['aptoken']),
            ], JSON_UNESCAPED_UNICODE);
            exit;
        }
        if ($op === 'wl') {
            $r = op_whitelist_list($ACCOUNT_ID, $DEST_ADDRESS);
            echo json_encode([
                'ok'          => $r['ok'],
                'http'        => $r['raw']['http'] ?? 0,
                'body'        => $r['raw']['body_raw'] ?? '',
                'verified'    => $r['verified'],
                'whitelist_id'=> $r['whitelist_id'],
                'provider_id' => $r['provider_id'],
                'has_token'   => !empty($_SESSION['aptoken']),
            ], JSON_UNESCAPED_UNICODE);
            exit;
        }
        if ($op === 'withdraw') {
            $amount  = $_POST['amount']  ?? '10';
            $variant = $_POST['variant'] ?? 'A';
            $otp     = $_POST['otp']     ?? null;

            $r = op_withdraw($ACCOUNT_ID, $DEFAULT_CURRENCY, $DEFAULT_NETWORK, $DEST_ADDRESS, $amount, $variant, $otp);

            echo json_encode([
                'ok'        => $r['ok'],
                'need_otp'  => $r['need_otp'] ?? false,
                'msg'       => $r['msg'] ?? null,
                'id'        => $r['id'] ?? null,
                'last_http' => $r['last']['raw']['http'] ?? 0,
                'last_body' => $r['last']['raw']['body_raw'] ?? '',
                'variant'   => $r['last']['variant'] ?? $variant,
                'payload'   => $r['last']['payload'] ?? null
            ], JSON_UNESCAPED_UNICODE);
            exit;
        }

        echo json_encode(['ok'=>false,'error'=>'op_desconocida']);
        exit;
    } catch (Throwable $e) {
        echo json_encode(['ok'=>false,'error'=>$e->getMessage()]);
        exit;
    }
}

/* =======================================================
 * ==================  [7] HTML/JS  ======================
 * ======================================================= */
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
$hasToken = !empty($_SESSION['aptoken']);
$tokenMasked = $hasToken ? (substr($_SESSION['aptoken'],0,8).'•••'.substr($_SESSION['aptoken'],-4)) : '—';
?>
<!doctype html>
<html lang="es">
<head>
<meta charset="utf-8">
<title>Rescate USDT_BSC · Async v2.8.1</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<style>
body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;background:#f6f8fb;margin:0;color:#223;}
.container{max-width:1000px;margin:0 auto;padding:20px}
.card{background:#fff;border:1px solid #e7ecf2;border-radius:10px;box-shadow:0 1px 3px rgba(15,23,42,.06);padding:16px;margin:14px 0}
h1{margin:0 0 6px}
small{color:#445}
.grid{display:grid;gap:12px}
.grid2{grid-template-columns:1fr 1fr}
button{background:#2563eb;color:#fff;border:none;border-radius:8px;padding:10px 14px;cursor:pointer;font-weight:600}
button.secondary{background:#64748b}
pre{white-space:pre-wrap;background:#0f172a;color:#e5e7eb;padding:12px;border-radius:8px;overflow:auto;font-size:.88rem}
.bad{color:#7f1d1d;background:#fee2e2;border:1px solid #fecaca;padding:8px;border-radius:8px}
.good{color:#065f46;background:#ecfdf5;border:1px solid #d1fae5;padding:8px;border-radius:8px}
.hint{color:#1e3a8a;background:#eff6ff;border:1px solid #bfdbfe;padding:8px;border-radius:8px}
label{font-size:.9rem}
input,select{width:100%;padding:8px;border:1px solid #cbd5e1;border-radius:8px}
.row{display:flex;gap:10px;align-items:center;flex-wrap:wrap}
</style>
</head>
<body>
<div class="container">
  <h1>Rescate USDT_BSC · Depósito huérfano → Whitelist → Retiro (Async v2.8.1)</h1>
  <small>Cuenta: <?=h($ACCOUNT_ID)?> · Moneda: USDT · Red: USDT_BSC</small>

  <div class="card">
    <h3>Paso 1: Obtener/renovar aptoken (asíncrono)</h3>
    <div class="row">
      <button id="btnAuth">Obtener/renovar aptoken</button>
      <div id="aptokenBox" class="<?= $hasToken?'good':'bad' ?>" style="min-width:280px">
        aptoken: <b id="aptokenText"><?=h($tokenMasked)?></b>
      </div>
    </div>
    <pre id="authLog" style="display:none"></pre>
  </div>

  <div class="card">
    <h3>Paso 2: Historial (últimas 5) · Async</h3>
    <div class="row">
      <button id="btnHist">Listar</button>
      <button id="btnAuthThenHist" class="secondary">Renovar aptoken + Listar</button>
    </div>
    <pre id="histOut"></pre>
  </div>

  <div class="card">
    <h3>Paso 3: Dirección de depósito USDT_BSC · Async</h3>
    <div class="row">
      <button id="btnDep">Consultar</button>
      <button id="btnAuthThenDep" class="secondary">Renovar aptoken + Consultar</button>
    </div>
    <pre id="depOut"></pre>
  </div>

  <div class="card">
    <h3>Paso 4 (pre): Verificar whitelist de destino · Async</h3>
    <div class="row">
      <button id="btnWL">Verificar</button>
      <button id="btnAuthThenWL" class="secondary">Renovar aptoken + Verificar</button>
    </div>
    <pre id="wlOut"></pre>
  </div>

  <div class="card">
    <h3>Paso 6: Retiro · Async</h3>
    <div class="grid grid2">
      <div>
        <label>Monto (USDT)</label>
        <input id="amount" type="text" value="10">
      </div>
      <div>
        <label>Variante</label>
        <select id="variant">
          <option value="A">A — USDT + USDT_BSC + address + whitelist_id + provider_id</option>
          <option value="B" selected>B — USDT + USDT_BSC + address + whitelist_id (sin provider_id)</option>
          <option value="C">C — USDT + USDT_BSC + solo address</option>
        </select>
      </div>
    </div>
    <div class="row" style="margin-top:10px">
      <input id="otp" type="text" placeholder="OTP 2FA (solo si el backend lo pide)" style="max-width:220px">
      <button id="btnWdraw">Enviar retiro</button>
      <button id="btnAuthThenWdraw" class="secondary">Renovar aptoken + Enviar</button>
    </div>
    <pre id="wdrawOut"></pre>
    <div class="hint" style="margin-top:8px">
      Si aparece <b>required_token (401)</b>, usa un OTP <b>nuevo</b> y reenvía la misma variante. Si aparece <b>internal_error (500)</b>, cambia solo una cosa (monto 8–12 USDT o usar/quitar provider_id) y reintenta.
    </div>
  </div>
</div>

<script>
const $ = sel => document.querySelector(sel);
function show(el, txt){ const n=$(el); n.style.display='block'; n.textContent = (typeof txt==='string')?txt:JSON.stringify(txt,null,2); }
function maskToken(t){ if(!t) return '—'; if(t.includes('•')) return t; return t.slice(0,8)+'•••'+t.slice(-4); }

async function call(op, method='GET', data=null){
  const opt = { method, headers: {} };
  if (method === 'POST'){
    opt.headers['Content-Type'] = 'application/x-www-form-urlencoded;charset=UTF-8';
    const form = new URLSearchParams();
    if (data) for (const k in data) form.append(k, data[k]);
    opt.body = form.toString();
  }
  const r = await fetch(`?api=1&op=${encodeURIComponent(op)}`, opt);
  const j = await r.json().catch(()=>({ok:false, http:r.status, body:'<parse error>'}));
  return j;
}

/* Paso 1 */
$('#btnAuth').addEventListener('click', async ()=>{
  const btn = $('#btnAuth');
  btn.disabled = true; btn.textContent = 'Autenticando...';
  try {
    const j = await call('auth','POST');
    show('#authLog', j);
    const ok = !!j.ok;
    $('#aptokenBox').className = ok ? 'good' : 'bad';
    $('#aptokenText').textContent = maskToken(j.token);
  } finally {
    setTimeout(()=>{ btn.disabled = false; btn.textContent = 'Obtener/renovar aptoken'; }, 800);
  }
});

/* Paso 2 */
$('#btnHist').addEventListener('click', async ()=>{
  const j = await call('hist','GET');
  show('#histOut', j);
});
$('#btnAuthThenHist').addEventListener('click', async ()=>{
  const a = await call('auth','POST');
  $('#aptokenBox').className = a.ok ? 'good' : 'bad';
  $('#aptokenText').textContent = maskToken(a.token);
  const j = await call('hist','GET');
  show('#histOut', j);
});

/* Paso 3 */
$('#btnDep').addEventListener('click', async ()=>{
  const j = await call('deposit','GET');
  show('#depOut', j);
});
$('#btnAuthThenDep').addEventListener('click', async ()=>{
  const a = await call('auth','POST');
  $('#aptokenBox').className = a.ok ? 'good' : 'bad';
  $('#aptokenText').textContent = maskToken(a.token);
  const j = await call('deposit','GET');
  show('#depOut', j);
});

/* Paso 4 */
$('#btnWL').addEventListener('click', async ()=>{
  const j = await call('wl','GET');
  show('#wlOut', j);
});
$('#btnAuthThenWL').addEventListener('click', async ()=>{
  const a = await call('auth','POST');
  $('#aptokenBox').className = a.ok ? 'good' : 'bad';
  $('#aptokenText').textContent = maskToken(a.token);
  const j = await call('wl','GET');
  show('#wlOut', j);
});

/* Paso 6 */
async function doWithdraw(flowAuthFirst){
  if (flowAuthFirst){
    const a = await call('auth','POST');
    $('#aptokenBox').className = a.ok ? 'good' : 'bad';
    $('#aptokenText').textContent = maskToken(a.token);
  }
  const amount = $('#amount').value.trim() || '10';
  const variant= $('#variant').value;
  const otp    = $('#otp').value.trim();
  const j = await call('withdraw','POST',{ amount, variant, otp });
  show('#wdrawOut', j);
}
$('#btnWdraw').addEventListener('click', ()=>doWithdraw(false));
$('#btnAuthThenWdraw').addEventListener('click', ()=>doWithdraw(true));
</script>
</body>
</html>
