<?php
// notbank_historial_withdraw.php
//
// Este script amplía la funcionalidad de consulta de transacciones para permitir
// seleccionar una transacción y crear un retiro de criptomonedas a una
// dirección especificada. Utiliza la API de Notbank para autenticarse,
// listar transacciones y enviar un retiro a la red blockchain a través del
// endpoint CreateCryptoWithdraw. Las direcciones de destino deben estar
// previamente whitelisteadas según la documentación de Notbank【261154067702217†screenshot】.

/*
 * ADVERTENCIA: El envío de fondos es una operación sensible. Este script es
 * solo un ejemplo educativo que demuestra cómo estructurar las llamadas a
 * la API. No ejecuta ninguna transacción automáticamente y asume que ya
 * existen direcciones whitelisteadas en la cuenta de origen. Pruebe en
 * entornos seguros y revise la documentación de Notbank antes de usarlo
 * en producción【261154067702217†screenshot】.
 */

// Iniciar la sesión para almacenar tokens persistentes entre solicitudes
// Nota: eliminamos el flujo de autenticación 2FA porque, según la
// documentación oficial de Notbank, el uso de claves API en HTTP
// anula el requisito de 2FA y la llamada AuthenticateUser devuelve
// directamente un SessionToken válido【305021109692464†L1159-L1223】. El
// token se almacena en la sesión para evitar repetidas autenticaciones.
session_start();

// ----- Configuración -----
$apiBaseUrl  = 'https://api.notbank.exchange';
$apiKey      = '40b266e4f2f010631ddfb26cda4d915a';    // Llave pública (APIKey)
$apiSecret   = '5bc92a8df76d2d88248590e0fef53ecc';    // Llave privada (secreta)
$userId      = 1081;                                   // ID de usuario
$accountId   = 4231;                                   // ID de cuenta. Sustituye con tu account_id real para filtrar mejor

/**
 * Genera la firma HMAC-SHA256 necesaria para la autenticación.
 */
function nb_generateSignature($nonce, $userId, $apiKey, $apiSecret) {
    $payload = $nonce . $userId . $apiKey;
    return hash_hmac('sha256', $payload, $apiSecret);
}

/**
 * Realiza la autenticación con Notbank y devuelve el SessionToken (aptoken).
 */
function nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId) {
    // Autenticación basada en nonce de milisegundos y cabecera Content-Type
    // Usamos un nonce de alta resolución (ms) como recomienda Notbank para evitar duplicidades
    $nonce = (string) round(microtime(true) * 1000);
    // La firma se genera concatenando nonce + userId + apiKey y aplicando HMAC-SHA256 con la clave secreta
    $signature = nb_generateSignature($nonce, $userId, $apiKey, $apiSecret);
    $url = rtrim($apiBaseUrl, '/') . '/AP/AuthenticateUser';
    $headers = [
        'APIKey: ' . $apiKey,
        'Signature: ' . $signature,
        'UserId: ' . $userId,
        'Nonce: ' . $nonce,
        // Incluir Content-Type como en el ejemplo original de autenticación
        'Content-Type: application/json'
    ];
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER     => $headers
    ]);
    $response = curl_exec($ch);
    if (curl_errno($ch)) {
        curl_close($ch);
        return false;
    }
    curl_close($ch);
    $data = json_decode($response, true);
    // Notbank puede devolver SessionToken en el nivel superior o dentro de 'data'
    if (is_array($data)) {
        if (isset($data['SessionToken']) && !empty($data['SessionToken'])) {
            return $data['SessionToken'];
        }
        if (isset($data['data']['SessionToken']) && !empty($data['data']['SessionToken'])) {
            return $data['data']['SessionToken'];
        }
        if (isset($data['sessionToken']) && !empty($data['sessionToken'])) {
            return $data['sessionToken'];
        }
    }
    return false;
}

/*
 * Eliminamos la función nb_authenticate2fa() ya que, según la
 * documentación de Notbank, la llamada AuthenticateUser con APIKey y
 * firma HMAC devuelve un token de sesión válido sin necesidad de un
 * flujo extra de autenticación de dos factores en HTTP【305021109692464†L1159-L1223】.
 *
 * Si tu cuenta tiene 2FA activado y la API requiere un código OTP
 * adicional, dicho código se debe incluir en el cuerpo del retiro
 * (parámetro 'otp'), tal como especifica el endpoint
 * CreateCryptoWithdraw【261154067702217†screenshot】. No existe una llamada HTTP
 * documentada para completar la autenticación 2FA separadamente.
 */

/**
 * Obtiene todas las transacciones de la cuenta especificada.
 *
 * @param string $apiBaseUrl  Base URL de la API
 * @param string $aptoken     Token de sesión
 * @param int    $accountId   ID de la cuenta
 * @param string|null $fromDate Fecha inicial (dd/mm/yyyy) opcional
 * @param string|null $toDate   Fecha final (dd/mm/yyyy) opcional
 * @return array               Listado de transacciones
 */
function nb_getAllTransactions($apiBaseUrl, $aptoken, $accountId, $fromDate = null, $toDate = null) {
    $page = 1;
    $pageSize = 50;
    $allData = [];
    do {
        $params = [
            'page'      => $page,
            'page_size' => $pageSize
        ];
        if ($accountId) {
            $params['accountId'] = $accountId;
        }
        if ($fromDate) {
            $params['from_date'] = $fromDate;
        }
        if ($toDate) {
            $params['to_date'] = $toDate;
        }
        $url = rtrim($apiBaseUrl, '/') . '/api/nb/wallet/transactions?' . http_build_query($params);
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER     => [
                'Content-Type: application/json',
                'aptoken: ' . $aptoken
            ]
        ]);
        $response = curl_exec($ch);
        if (curl_errno($ch)) {
            curl_close($ch);
            break;
        }
        curl_close($ch);
        $result = json_decode($response, true);
        $data = $result['data'] ?? [];
        $allData = array_merge($allData, $data);
        $page++;
        $hasMore = count($data) >= $pageSize;
    } while ($hasMore);
    return $allData;
}

/**
 * Envía una solicitud de retiro de criptomonedas (CreateCryptoWithdraw).
 *
 * @param string $apiBaseUrl   URL base de Notbank
 * @param string $aptoken      Token de sesión
 * @param int    $accountId    ID de la cuenta origen
 * @param string $currency     Código de la moneda (p.ej. "USDT", "BTC")
 * @param string $network      Red de la moneda (p.ej. "USDT_BSC")
 * @param string $address      Dirección destino
 * @param float  $amount       Monto a retirar
 * @param string|null $memoTag Memo o tag (si la red lo requiere)
 * @param string|null $otp     Código de autenticación en dos pasos
 * @return array|false         Respuesta de la API decodificada o false en caso de error
 */
function nb_createCryptoWithdraw($apiBaseUrl, $aptoken, $accountId, $currency, $network, $address, $amount, $memoTag = null, $otp = null) {
    $url = rtrim($apiBaseUrl, '/') . '/api/nb/wallet/crypto/withdrawal';
    $payload = [
        'account_id' => (int)$accountId,
        'currency'   => $currency,
        'network'    => $network,
        'address'    => $address,
        'amount'     => $amount
    ];
    if (!empty($memoTag)) {
        $payload['memo_or_tag'] = $memoTag;
    }
    if (!empty($otp)) {
        $payload['otp'] = $otp;
    }
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/json',
            'aptoken: ' . $aptoken
        ],
        CURLOPT_POSTFIELDS     => json_encode($payload)
    ]);
    $response = curl_exec($ch);
    if (curl_errno($ch)) {
        curl_close($ch);
        return false;
    }
    curl_close($ch);
    return json_decode($response, true);
}

/**
 * Devuelve la lista de direcciones whitelisteadas para una cuenta dada.
 *
 * Este endpoint corresponde a GET /api/nb/wallet/crypto/whitelist-addresses
 * y devuelve un listado de objetos con los campos id, currency, label,
 * network, address, memo, verified, provider_id. El parámetro
 * account_id es obligatorio según la documentación de Notbank【56026549371736†screenshot】.
 *
 * @param string $apiBaseUrl  Base URL de la API
 * @param string $aptoken     Token de sesión
 * @param int    $accountId   ID de la cuenta cuyos whitelists se desean listar
 * @param string|null $search  Texto opcional para filtrar resultados por moneda, etiqueta o dirección
 * @return array|false         Lista de direcciones whitelisteadas o false en caso de error
 */
function nb_getWhitelistedAddresses($apiBaseUrl, $aptoken, $accountId, $search = null) {
    if (!$accountId) {
        return [];
    }
    // Construir la URL con parámetros de consulta
    $params = ['account_id' => (int)$accountId];
    if (!empty($search)) {
        $params['search'] = $search;
    }
    $url = rtrim($apiBaseUrl, '/') . '/api/nb/wallet/crypto/whitelist-addresses?' . http_build_query($params);
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'aptoken: ' . $aptoken
        ]
    ]);
    $response = curl_exec($ch);
    if (curl_errno($ch)) {
        curl_close($ch);
        return false;
    }
    curl_close($ch);
    $result = json_decode($response, true);
    // Verificar que la respuesta sea un array y tenga status success
    if (is_array($result) && isset($result['status']) && strtolower($result['status']) === 'success') {
        return $result['data'] ?? [];
    }
    return false;
}

/**
 * Comprueba si una dirección está whitelisteada y verificada.
 *
 * Recorre el listado de whitelists devuelto por nb_getWhitelistedAddresses() y
 * compara tanto la dirección (sin distinción de mayúsculas) como la red y la
 * moneda. Solo se considera válida si el campo "verified" es true.
 *
 * @param array  $whitelist  Lista de direcciones whitelisteadas
 * @param string $address    Dirección que se desea verificar
 * @param string $network    Red a la que pertenece la dirección
 * @param string $currency   Moneda de la dirección
 * @return bool              True si la dirección está confirmada en la whitelist, false en caso contrario
 */
function nb_isAddressWhitelisted(array $whitelist, $address, $network, $currency) {
    $address = strtolower(trim($address));
    foreach ($whitelist as $entry) {
        // Normalizar datos de la entrada
        $entryAddress = isset($entry['address']) ? strtolower(trim($entry['address'])) : '';
        $verified     = isset($entry['verified']) ? (bool)$entry['verified'] : false;
        if ($verified && $entryAddress === $address) {
            // Una coincidencia por dirección es suficiente, independientemente de red o moneda.
            return true;
        }
    }
    return false;
}

/**
 * Añade una dirección a la lista blanca (whitelist) mediante el endpoint
 * AddWhitelistedAddress. Se requiere un token de sesión válido y el permiso
 * NotbankWithdraw【305021109692464†L110-L124】.
 *
 * @param string $apiBaseUrl Base URL de la API
 * @param string $aptoken    Token de sesión
 * @param int    $accountId  ID de la cuenta para la cual se crea la dirección
 * @param string $currency   Moneda (ej. "USDT")
 * @param string $network    Red (ej. "USDT_BSC")
 * @param string $address    Dirección a whitelistear
 * @param string $label      Nombre o etiqueta para identificar la dirección
 * @param string|null $memo  Memo o tag opcional según la red
 * @param string $otp        Código de autenticación de 2FA
 * @return array|false       Respuesta decodificada o false en caso de error
 */
function nb_addWhitelistedAddress($apiBaseUrl, $aptoken, $accountId, $currency, $network, $address, $label, $memo = null, $otp) {
    $payload = [
        'account_id' => (int)$accountId,
        'currency'   => $currency,
        'network'    => $network,
        'address'    => $address,
        'label'      => $label,
        'otp'        => $otp
    ];
    if (!empty($memo)) {
        $payload['memo_or_tag'] = $memo;
    }
    $url = rtrim($apiBaseUrl, '/') . '/api/nb/wallet/crypto/whitelist-addresses';
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/json',
            'aptoken: ' . $aptoken
        ],
        CURLOPT_POSTFIELDS     => json_encode($payload)
    ]);
    $response = curl_exec($ch);
    if (curl_errno($ch)) {
        curl_close($ch);
        return false;
    }
    curl_close($ch);
    return json_decode($response, true);
}

/**
 * Confirma una dirección whitelisteada mediante el endpoint
 * ConfirmWhitelistedAddress. Debes proporcionar el ID devuelto al añadir la
 * dirección y el código SMS recibido por tu proveedor【305021109692464†L110-L124】.
 *
 * @param string $apiBaseUrl          Base URL de la API
 * @param string $aptoken             Token de sesión
 * @param string $whitelistedAddressId ID de la dirección whitelisteada
 * @param int    $accountId           ID de la cuenta
 * @param string $smsCode             Código SMS enviado para confirmar
 * @return array|false                Respuesta decodificada o false en caso de error
 */
function nb_confirmWhitelistedAddress($apiBaseUrl, $aptoken, $whitelistedAddressId, $accountId, $smsCode) {
    $url = rtrim($apiBaseUrl, '/') . '/api/nb/wallet/crypto/whitelist-addresses/' . $whitelistedAddressId . '/verification';
    $payload = [
        'account_id' => (int)$accountId,
        'sms_code'   => $smsCode
    ];
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/json',
            'aptoken: ' . $aptoken
        ],
        CURLOPT_POSTFIELDS     => json_encode($payload)
    ]);
    $response = curl_exec($ch);
    if (curl_errno($ch)) {
        curl_close($ch);
        return false;
    }
    curl_close($ch);
    return json_decode($response, true);
}

/**
 * Maneja la respuesta de un retiro y determina si el error está relacionado con
 * autenticación o 2FA. Devuelve un mensaje para mostrar al usuario y un
 * indicador de si se debería reintentar la llamada tras reautenticar.
 *
 * @param array|null $response  Respuesta de la API
 * @return array                [mensaje, reintentar]
 */
function nb_handleWithdrawResponse($response) {
    if (!is_array($response)) {
        return [
            'Error al realizar el retiro: respuesta inesperada del servidor.',
            false
        ];
    }
    // Éxito
    if (isset($response['status']) && strtolower($response['status']) === 'success') {
        $id = isset($response['data']) ? $response['data'] : '';
        return ['Retiro solicitado con éxito. ID de solicitud: ' . htmlspecialchars($id), false];
    }
    // Verificar códigos o mensajes comunes de error
    $errMsg = '';
    if (isset($response['message']) && is_string($response['message'])) {
        $errMsg = $response['message'];
    } elseif (isset($response['detail']) && is_string($response['detail'])) {
        $errMsg = $response['detail'];
    } elseif (isset($response['error']) && is_string($response['error'])) {
        $errMsg = $response['error'];
    } elseif (isset($response['errormsg']) && is_string($response['errormsg'])) {
        $errMsg = $response['errormsg'];
    }
    $reauthenticate = false;
    // Si el error sugiere problemas de autenticación o 2FA, instruir al usuario
    if (preg_match('/unauthorized/i', $errMsg) || preg_match('/sin autoriz/iu', $errMsg) || preg_match('/token/i', $errMsg) || preg_match('/auth/i', $errMsg)) {
        // Acceso no autorizado o token inválido: reintentar autenticación
        $reauthenticate = true;
        $mensaje = 'Sesión expirada o inválida. Se intentará reautenticar. Si el problema persiste, comprueba que tu usuario tenga el permiso NotbankWithdraw y que la dirección esté en la lista blanca.';
    } elseif (preg_match('/2FA/i', $errMsg) || preg_match('/two factor/i', $errMsg) || preg_match('/otp/i', $errMsg)) {
        // Error de 2FA: puede ser código incorrecto o expirado
        $mensaje = 'Falló la autenticación de dos factores. Verifica que el código de tu aplicación esté vigente y vuelve a intentarlo.';
    } elseif (stripos($errMsg, 'not authorized') !== false || stripos($errMsg, 'acceso no autorizado') !== false) {
        // Mensaje explícito de no autorizado
        $mensaje = 'Acceso no autorizado: verifica que tu API Key tenga habilitado el permiso NotbankWithdraw y que la cuenta cumpla los requisitos de retiro.';
    } else {
        // Mensaje genérico
        $mensaje = 'Error al realizar el retiro: ' . htmlspecialchars($errMsg);
    }
    return [$mensaje, $reauthenticate];
}

// ------ Lógica del script ------
// Recoger filtros de fecha enviados por el usuario
$fromDate     = isset($_POST['fromDate']) && trim($_POST['fromDate']) !== '' ? trim($_POST['fromDate']) : null;
$toDate       = isset($_POST['toDate'])   && trim($_POST['toDate'])   !== '' ? trim($_POST['toDate'])   : null;
$transactions = [];
$message      = '';

// A partir de esta versión, eliminamos el flujo de autenticación 2FA. El token de sesión
// se obtiene directamente mediante nb_authenticate(). Si el retiro necesita
// autenticación de dos factores, el código se pasa como parámetro "otp" en
// nb_createCryptoWithdraw()【261154067702217†screenshot】. La sesión almacena el
// SessionToken para reutilizarlo.

// Paso 1: manejar las acciones enviadas por el usuario
// Recoger la acción (si existe) de la solicitud POST
$action = isset($_POST['action']) ? $_POST['action'] : null;
if ($action) {
    switch ($action) {
        case 'withdraw':
            // Recoger datos del formulario de retiro
            $withdrawAccountId = (int)($_POST['source_account_id'] ?? 0);
            $currency   = $_POST['currency'] ?? '';
            $amount     = (float) ($_POST['amount'] ?? 0);
            $network    = $_POST['network'] ?? '';
            $address    = $_POST['address'] ?? '';
            $memoTag    = $_POST['memo'] ?? null;
            $withdrawOtp= $_POST['otp'] ?? null;
            if ($withdrawAccountId && $currency && $amount > 0 && $network && $address) {
                // Obtener el token de sesión: si existe en la sesión, usarlo; si no, autenticar de nuevo
                if (isset($_SESSION['aptoken']) && $_SESSION['aptoken']) {
                    $aptoken = $_SESSION['aptoken'];
                } else {
                    $aptoken = nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId);
                    if ($aptoken) {
                        $_SESSION['aptoken'] = $aptoken;
                    }
                }
                if ($aptoken) {
                    // Verificar que la dirección esté whitelisteada
                    $whitelist = nb_getWhitelistedAddresses($apiBaseUrl, $aptoken, $withdrawAccountId);
                    if ($whitelist === false) {
                        $message = 'No se pudo obtener la lista de direcciones whitelisteadas. Asegúrate de que tu cuenta tenga habilitado el permiso NotbankWithdraw.';
                    } else {
                        if (!nb_isAddressWhitelisted($whitelist, $address, $network, $currency)) {
                            $addressClean = htmlspecialchars($address);
                            $message = 'La dirección ' . $addressClean . ' no está en la whitelist o no ha sido confirmada. Agrégala a la whitelist desde la plataforma o mediante el formulario siguiente antes de intentar el retiro.';
                        } else {
                            // La dirección está verificada: realizar el retiro
                            $resp = nb_createCryptoWithdraw($apiBaseUrl, $aptoken, $withdrawAccountId, $currency, $network, $address, $amount, $memoTag, $withdrawOtp);
                            list($msg, $reauth) = nb_handleWithdrawResponse($resp);
                            if ($reauth) {
                                // Reautenticar y reintentar una vez
                                unset($_SESSION['aptoken']);
                                $aptoken = nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId);
                                if ($aptoken) {
                                    $_SESSION['aptoken'] = $aptoken;
                                    $respRetry = nb_createCryptoWithdraw($apiBaseUrl, $aptoken, $withdrawAccountId, $currency, $network, $address, $amount, $memoTag, $withdrawOtp);
                                    list($msg, $reauth2) = nb_handleWithdrawResponse($respRetry);
                                    $message = $msg;
                                } else {
                                    $message = 'No se pudo reautenticar. Por favor vuelve a cargar la página e intenta de nuevo.';
                                }
                            } else {
                                $message = $msg;
                            }
                        }
                    }
                } else {
                    $message = 'Error de autenticación al obtener el token para el retiro.';
                }
            } else {
                $message = 'Completa todos los campos obligatorios para el retiro.';
            }
            break;
        case 'add_whitelist':
            // Recoger datos del formulario para agregar una dirección
            $wlAccountId = (int)($_POST['wl_account_id'] ?? 0);
            $wlCurrency  = $_POST['wl_currency'] ?? '';
            $wlNetwork   = $_POST['wl_network'] ?? '';
            $wlAddress   = $_POST['wl_address'] ?? '';
            $wlLabel     = $_POST['wl_label'] ?? '';
            $wlMemo      = $_POST['wl_memo'] ?? null;
            $wlOtp       = $_POST['wl_otp'] ?? '';
            if ($wlAccountId && $wlCurrency && $wlNetwork && $wlAddress && $wlLabel && $wlOtp) {
                if (isset($_SESSION['aptoken']) && $_SESSION['aptoken']) {
                    $aptokenWL = $_SESSION['aptoken'];
                } else {
                    $aptokenWL = nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId);
                    if ($aptokenWL) {
                        $_SESSION['aptoken'] = $aptokenWL;
                    }
                }
                if ($aptokenWL) {
                    $respAdd = nb_addWhitelistedAddress($apiBaseUrl, $aptokenWL, $wlAccountId, $wlCurrency, $wlNetwork, $wlAddress, $wlLabel, $wlMemo, $wlOtp);
                    if (is_array($respAdd) && isset($respAdd['status']) && strtolower($respAdd['status']) === 'success') {
                        $newId = $respAdd['data']['id'] ?? '';
                        $message = 'Dirección whitelisteada creada con éxito. ID: ' . htmlspecialchars($newId) . '. Ahora debes confirmarla con el código SMS usando el formulario de confirmación.';
                    } else {
                        $errMsgWL = '';
                        if (isset($respAdd['message'])) $errMsgWL = $respAdd['message'];
                        elseif (isset($respAdd['detail'])) $errMsgWL = $respAdd['detail'];
                        elseif (isset($respAdd['error'])) $errMsgWL = $respAdd['error'];
                        $message = 'Error al crear la dirección whitelisteada: ' . htmlspecialchars($errMsgWL);
                    }
                } else {
                    $message = 'No se pudo autenticar para agregar la dirección.';
                }
            } else {
                $message = 'Completa todos los campos obligatorios para agregar una dirección a la whitelist.';
            }
            break;
        case 'confirm_whitelist':
            // Recoger datos del formulario de confirmación
            $confirmId       = $_POST['confirm_id'] ?? '';
            $confirmCode     = $_POST['confirm_code'] ?? '';
            $confirmAccountId = (int)($_POST['confirm_account_id'] ?? 0);
            if ($confirmId && $confirmCode && $confirmAccountId) {
                if (isset($_SESSION['aptoken']) && $_SESSION['aptoken']) {
                    $aptokenConfirm = $_SESSION['aptoken'];
                } else {
                    $aptokenConfirm = nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId);
                    if ($aptokenConfirm) {
                        $_SESSION['aptoken'] = $aptokenConfirm;
                    }
                }
                if ($aptokenConfirm) {
                    $respConfirm = nb_confirmWhitelistedAddress($apiBaseUrl, $aptokenConfirm, $confirmId, $confirmAccountId, $confirmCode);
                    if (is_array($respConfirm) && isset($respConfirm['status']) && strtolower($respConfirm['status']) === 'success') {
                        $message = 'Dirección confirmada con éxito. Ya puedes utilizarla para retirar fondos.';
                    } else {
                        $errMsgC = '';
                        if (isset($respConfirm['message'])) $errMsgC = $respConfirm['message'];
                        elseif (isset($respConfirm['detail'])) $errMsgC = $respConfirm['detail'];
                        elseif (isset($respConfirm['error'])) $errMsgC = $respConfirm['error'];
                        $message = 'Error al confirmar la dirección: ' . htmlspecialchars($errMsgC);
                    }
                } else {
                    $message = 'No se pudo autenticar para confirmar la dirección.';
                }
            } else {
                $message = 'Proporciona el ID de la dirección y el código SMS para confirmarla.';
            }
            break;
        case 'check_whitelist':
            // Recoger datos del formulario de verificación
            $checkAccountId = (int)($_POST['check_account_id'] ?? 0);
            $checkCurrency  = $_POST['check_currency'] ?? '';
            $checkNetwork   = $_POST['check_network'] ?? '';
            $checkAddress   = $_POST['check_address'] ?? '';
            if ($checkAccountId && $checkAddress) {
                if (isset($_SESSION['aptoken']) && $_SESSION['aptoken']) {
                    $aptokenCheck = $_SESSION['aptoken'];
                } else {
                    $aptokenCheck = nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId);
                    if ($aptokenCheck) {
                        $_SESSION['aptoken'] = $aptokenCheck;
                    }
                }
                if ($aptokenCheck) {
                    $wlList = nb_getWhitelistedAddresses($apiBaseUrl, $aptokenCheck, $checkAccountId);
                    if ($wlList === false) {
                        $message = 'No se pudo obtener la lista de direcciones whitelisteadas. Asegúrate de que tu API Key tenga los permisos necesarios.';
                    } else {
                        $isWhitelisted = nb_isAddressWhitelisted($wlList, $checkAddress, $checkNetwork, $checkCurrency);
                        $addrDisplay = htmlspecialchars($checkAddress);
                        if ($isWhitelisted) {
                            $message = 'La dirección ' . $addrDisplay . ' está en la whitelist de la cuenta y ha sido verificada.';
                        } else {
                            $message  = 'La dirección ' . $addrDisplay . ' NO está en la whitelist o no ha sido confirmada.';
                            $message .= ' Ten en cuenta que las direcciones de depósito (donde recibes fondos en tu cuenta Notbank) no forman parte de la whitelist de retiros.';
                            $message .= ' Solo las direcciones a las que enviarás fondos deben añadirse y confirmarse como whitelisteadas.';
                        }
                    }
                } else {
                    $message = 'No se pudo autenticar para comprobar la whitelist.';
                }
            } else {
                $message = 'Proporciona al menos la cuenta y la dirección para verificar en la whitelist.';
            }
            break;
        default:
            // Acción no reconocida; no hacemos nada
            break;
    }
}

// Paso 2: recuperar transacciones para mostrar
// Utilizar aptoken de la sesión si existe, de lo contrario autenticar de nuevo
if (isset($_SESSION['aptoken']) && $_SESSION['aptoken']) {
    $aptokenForList = $_SESSION['aptoken'];
} else {
    $aptokenForList = nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId);
    if ($aptokenForList) {
        $_SESSION['aptoken'] = $aptokenForList;
    }
}
if ($aptokenForList) {
    $transactions = nb_getAllTransactions($apiBaseUrl, $aptokenForList, $accountId, $fromDate ?: null, $toDate ?: null);
    // Ordenar las transacciones por fecha descendente (si existe el campo created_at)
    usort($transactions, function ($a, $b) {
        $dateA = isset($a['created_at']) ? strtotime($a['created_at']) : 0;
        $dateB = isset($b['created_at']) ? strtotime($b['created_at']) : 0;
        return $dateB <=> $dateA;
    });
    // Limitar a las 5 transacciones más recientes para evitar desbordes de tabla
    $transactions = array_slice($transactions, 0, 5);
} else {
    $message = $message ?: 'No se pudo autenticar para obtener las transacciones.';
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Historial y retiros de criptomonedas - Notbank</title>
<style>
        /* Diseño general */
        body {
            font-family: "Segoe UI", Tahoma, sans-serif;
            margin: 0;
            padding: 0;
            background-color: #f5f7fa;
            color: #333;
        }
        .container {
            max-width: 1000px;
            margin: 0 auto;
            padding: 20px;
        }
        h1 {
            margin-top: 0;
            color: #2a3f54;
        }
        p.description {
            color: #556871;
            margin-bottom: 20px;
        }
        /* Tarjetas */
        .card {
            background-color: #fff;
            border-radius: 8px;
            box-shadow: 0 2px 6px rgba(0, 0, 0, 0.05);
            padding: 20px;
            margin-bottom: 20px;
        }
        /* Formularios */
        form {
            margin-bottom: 0;
        }
        .form-row {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
        }
        .form-group {
            flex: 1;
            min-width: 180px;
            display: flex;
            flex-direction: column;
        }
        label {
            font-size: 0.85rem;
            margin-bottom: 4px;
            color: #3a3a3a;
        }
        input[type="text"],
        input[type="number"],
        input[type="password"],
        select {
            padding: 8px 10px;
            border: 1px solid #ccc;
            border-radius: 4px;
            font-size: 0.9rem;
        }
        button {
            padding: 8px 16px;
            background-color: #5b9bd5;
            border: none;
            border-radius: 4px;
            color: white;
            font-weight: bold;
            cursor: pointer;
            transition: background-color 0.2s ease;
        }
        button:hover {
            background-color: #4a86c5;
        }
        /* Mensajes */
        .message {
            margin-top: 20px;
            padding: 12px 16px;
            border-radius: 4px;
        }
        .message.success {
            background-color: #e0f5e9;
            color: #0a572c;
        }
        .message.error {
            background-color: #fbeaea;
            color: #a94442;
        }
        /* Tabla */
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 10px;
        }
        th, td {
            padding: 10px;
            border-bottom: 1px solid #eee;
            font-size: 0.9rem;
        }
        th {
            background-color: #f0f3f6;
            color: #445364;
            text-align: left;
            font-weight: 600;
        }
        tr:nth-child(even) td {
            background-color: #fafcfe;
        }
        tr:hover td {
            background-color: #eef4fc;
        }
        /* Formulario de retiro dentro de la tabla */
        .withdraw-form {
            background-color: #f6f8fa;
            border: 1px solid #e1e7ef;
            border-radius: 4px;
            padding: 8px;
        }
        .withdraw-form label {
            margin-bottom: 2px;
            font-size: 0.8rem;
            color: #445364;
        }
        .withdraw-form input[type="text"],
        .withdraw-form input[type="number"] {
            padding: 6px;
            border: 1px solid #ccd5e0;
            border-radius: 4px;
            font-size: 0.8rem;
        }
        .withdraw-form button {
            margin-top: 6px;
            width: 100%;
            font-size: 0.9rem;
        }

        /* Lista de transacciones en formato tarjetas */
        .transaction-list {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 16px;
        }
        .tx-item {
            display: flex;
            flex-direction: row;
            flex-wrap: wrap;
            gap: 16px;
            border: 1px solid #e1e7ef;
            border-radius: 8px;
            padding: 16px;
            background-color: #fafcfe;
            width: 100%;
            max-width: 700px;
        }
        .tx-info {
            flex: 1 1 250px;
            min-width: 200px;
        }
        .tx-form {
            flex: 1 1 250px;
            min-width: 200px;
        }
        .tx-info p {
            margin: 4px 0;
            font-size: 0.85rem;
            color: #445364;
        }
    </style>
</head>
<body>
    <div class="container">
<h1>Historial y retiros de criptomonedas</h1>
        <p class="description">Consulta los movimientos de tu cuenta y solicita retiros de criptomonedas. Recuerda que las direcciones de destino deben estar whitelisteadas. Si tu cuenta tiene 2FA, introduce el código de tu aplicación de autenticación cuando hagas el retiro (campo OTP).</p>

        <!-- La autenticación se realiza automáticamente usando las claves API. Si se requiere 2FA para los retiros, se debe indicar en el campo OTP del formulario de retiro. -->

        <!-- Tarjeta de filtros de historial -->
        <div class="card">
            <h2>Filtrar historial</h2>
            <form method="post">
                <div class="form-row">
                    <div class="form-group">
                        <label for="fromDate">Fecha desde (dd/mm/yyyy)</label>
                        <input type="text" id="fromDate" name="fromDate" value="<?php echo htmlspecialchars($fromDate ?? ''); ?>" placeholder="Opcional">
                    </div>
                    <div class="form-group">
                        <label for="toDate">Fecha hasta (dd/mm/yyyy)</label>
                        <input type="text" id="toDate" name="toDate" value="<?php echo htmlspecialchars($toDate ?? ''); ?>" placeholder="Opcional">
                    </div>
                    <div class="form-group" style="align-self:flex-end;">
                        <button type="submit">Buscar</button>
                    </div>
                </div>
            </form>
        </div>

        <!-- Mostrar mensaje si existe -->
        <?php if (!empty($message)): ?>
        <div class="message <?php echo (strpos($message, 'éxito') !== false ? 'success' : 'error'); ?>">
            <?php echo $message; ?>
        </div>
        <?php endif; ?>

        <!-- Tarjeta de resultados -->
        <div class="card">
            <h2>Transacciones encontradas</h2>
            <?php if (!empty($transactions)): ?>
            <!-- Lista de transacciones como tarjetas -->
            <div class="transaction-list">
                <?php foreach ($transactions as $tx): ?>
                <?php
                    // Preparar datos para la transacción y el formulario
                    $txDate    = htmlspecialchars($tx['created_at'] ?? '');
                    $txCurrencyRaw = $tx['currency'] ?? '';
                    $txAddress = htmlspecialchars($tx['address'] ?? '');
                    $txAmount  = $tx['amount'] ?? '';
                    $txType    = (isset($tx['direction']) && $tx['direction'] == 1) ? 'Entrada' : 'Salida';
                    $txHash    = htmlspecialchars($tx['hash'] ?? '');
                    // Determine default network and currency (split e.g. USDTBSC -> USDT_BSC)
                    $defaultNetwork = '';
                    $txCurrency = $txCurrencyRaw;
                    if (!empty($txCurrencyRaw) && preg_match('/([A-Za-z]+)(BSC|ETH|TRC|SOL)/', $txCurrencyRaw, $m)) {
                        $defaultNetwork = $m[1] . '_' . $m[2];
                        $txCurrency = $m[1];
                    }
                    // Fallback: if network couldn't be deduced, leave blank and let user fill
                ?>
                <div class="tx-item">
                    <!-- Bloque de información -->
                    <div class="tx-info">
                        <p><strong>Fecha:</strong> <?php echo $txDate; ?></p>
                        <p><strong>Moneda:</strong> <?php echo htmlspecialchars($txCurrencyRaw); ?></p>
                        <p><strong>Dirección:</strong> <?php echo $txAddress; ?></p>
                        <p><strong>Monto:</strong> <?php echo htmlspecialchars($txAmount); ?></p>
                        <p><strong>Tipo:</strong> <?php echo $txType; ?></p>
                        <p><strong>Hash:</strong> <?php echo $txHash; ?></p>
                    </div>
                    <!-- Bloque del formulario de retiro -->
                    <div class="tx-form">
                        <form method="post" class="withdraw-form">
                            <input type="hidden" name="action" value="withdraw">
                            <input type="hidden" name="source_account_id" value="<?php echo (int)($tx['account_id'] ?? $accountId); ?>">
                            <input type="hidden" name="currency" value="<?php echo htmlspecialchars($txCurrency); ?>">
                            <div class="form-group">
                                <label>Monto a retirar</label>
                                <input type="number" step="0.00000001" name="amount" value="<?php echo htmlspecialchars($txAmount); ?>" required>
                            </div>
                            <div class="form-group">
                                <label>Red</label>
                                <input type="text" name="network" value="<?php echo htmlspecialchars($defaultNetwork); ?>" required>
                            </div>
                            <div class="form-group">
                                <label>Dirección destino</label>
                                <input type="text" name="address" placeholder="0x..." required>
                            </div>
                            <div class="form-group">
                                <label>Memo/Tag (opcional)</label>
                                <input type="text" name="memo" placeholder="">
                            </div>
                            <div class="form-group">
                                <label>OTP (opcional)</label>
                                <input type="text" name="otp" placeholder="">
                            </div>
                            <button type="submit">Enviar</button>
                        </form>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php else: ?>
            <p>No se encontraron transacciones para los filtros seleccionados.</p>
            <?php endif; ?>
        </div>

        <!-- Tarjeta de administración de whitelist -->
        <div class="card">
            <h2>Administrar whitelist</h2>
            <p class="description">Agrega una nueva dirección a tu whitelist o confirma una dirección pendiente utilizando el código SMS que recibas. Todas las operaciones requieren permisos de retiro.</p>
            <!-- Formulario para agregar nueva dirección -->
            <h3>Agregar dirección</h3>
            <form method="post" class="withdraw-form" style="max-width:600px;">
                <input type="hidden" name="action" value="add_whitelist">
                <div class="form-row">
                    <div class="form-group">
                        <label>Cuenta ID</label>
                        <input type="number" name="wl_account_id" value="<?php echo (int)$accountId; ?>" required>
                    </div>
                    <div class="form-group">
                        <label>Moneda</label>
                        <input type="text" name="wl_currency" placeholder="USDT, BTC" required>
                    </div>
                    <div class="form-group">
                        <label>Red</label>
                        <input type="text" name="wl_network" placeholder="USDT_BSC, BTC_MAINNET" required>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Dirección</label>
                        <input type="text" name="wl_address" placeholder="0x... o dirección de la red" required>
                    </div>
                    <div class="form-group">
                        <label>Etiqueta</label>
                        <input type="text" name="wl_label" placeholder="Nombre descriptivo" required>
                    </div>
                    <div class="form-group">
                        <label>Memo/Tag (opcional)</label>
                        <input type="text" name="wl_memo" placeholder="">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>OTP / 2FA</label>
                        <input type="text" name="wl_otp" placeholder="Código 2FA" required>
                    </div>
                    <div class="form-group" style="align-self:flex-end;">
                        <button type="submit">Crear dirección</button>
                    </div>
                </div>
            </form>
            <!-- Formulario para confirmar dirección -->
            <h3>Confirmar dirección</h3>
            <form method="post" class="withdraw-form" style="max-width:600px;">
                <input type="hidden" name="action" value="confirm_whitelist">
                <div class="form-row">
                    <div class="form-group">
                        <label>ID de dirección whitelisteada</label>
                        <input type="text" name="confirm_id" placeholder="ID devuelto al crear" required>
                    </div>
                    <div class="form-group">
                        <label>Cuenta ID</label>
                        <input type="number" name="confirm_account_id" value="<?php echo (int)$accountId; ?>" required>
                    </div>
                    <div class="form-group">
                        <label>Código SMS</label>
                        <input type="text" name="confirm_code" placeholder="Código SMS" required>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group" style="align-self:flex-end;">
                        <button type="submit">Confirmar dirección</button>
                    </div>
                </div>
            </form>

            <!-- Formulario para consultar si una dirección está en la whitelist -->
            <h3>Consultar dirección whitelisteada</h3>
            <form method="post" class="withdraw-form" style="max-width:600px;">
                <input type="hidden" name="action" value="check_whitelist">
                <div class="form-row">
                    <div class="form-group">
                        <label>Cuenta ID</label>
                        <input type="number" name="check_account_id" value="<?php echo (int)$accountId; ?>" required>
                    </div>
                    <div class="form-group">
                        <label>Moneda (opcional)</label>
                        <input type="text" name="check_currency" placeholder="USDT, BTC">
                    </div>
                    <div class="form-group">
                        <label>Red (opcional)</label>
                        <input type="text" name="check_network" placeholder="USDT_BSC, BTC_MAINNET">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Dirección</label>
                        <input type="text" name="check_address" placeholder="0x... o dirección" required>
                    </div>
                    <div class="form-group" style="align-self:flex-end;">
                        <button type="submit">Verificar</button>
                    </div>
                </div>
            </form>
        </div>
    </div>
</body>
</html>