<?php
// notbank_historial_withdraw.php
//
// Historial + retiros Notbank con diagnóstico detallado.
// - Valida que la dirección esté whitelisteada **para la misma moneda y red**.
// - Normaliza redes: BEP20→BSC, ERC20→ETH, TRC20→TRC, y formatea {CUR}_{RED} (p.ej. USDT_BSC).
// - Evita floats en "amount" y lo envía como string decimal.
// - Captura http_code, headers (incl. x-request-id), body crudo y errors{} del JSON.
// - Muestra <details> con diagnóstico cuando algo falla, para saber EXACTAMENTE qué falta.
//
// Seguridad: este ejemplo imprime información de diagnóstico en pantalla.
// No dejes el diagnóstico activado en producción.

// --- Debug opcional (descomenta en pruebas locales) ---
// ini_set('display_errors', 1);
// ini_set('display_startup_errors', 1);
// error_reporting(E_ALL);

session_start();

// ----- Configuración -----
$apiBaseUrl  = 'https://api.notbank.exchange';
$apiKey      = '40b266e4f2f010631ddfb26cda4d915a'; // Llave pública (ejemplo del usuario)
$apiSecret   = '5bc92a8df76d2d88248590e0fef53ecc'; // Llave privada (ejemplo del usuario)
$userId      = 1081;                                // ID de usuario
$accountId   = 4231;                                // ID de cuenta por defecto

/**
 * Firma HMAC-SHA256 para AuthenticateUser.
 */
function nb_generateSignature($nonce, $userId, $apiKey, $apiSecret) {
    $payload = $nonce . $userId . $apiKey;
    return hash_hmac('sha256', $payload, $apiSecret);
}

/**
 * Autentica y devuelve SessionToken (aptoken).
 */
function nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId) {
    $nonce = (string) round(microtime(true) * 1000);
    $signature = nb_generateSignature($nonce, $userId, $apiKey, $apiSecret);
    $url = rtrim($apiBaseUrl, '/') . '/AP/AuthenticateUser';
    $headers = [
        'APIKey: ' . $apiKey,
        'Signature: ' . $signature,
        'UserId: ' . $userId,
        'Nonce: ' . $nonce,
        'Content-Type: application/json',
        'Accept: application/json'
    ];
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER     => $headers
    ]);
    $response = curl_exec($ch);
    if (curl_errno($ch)) { curl_close($ch); return false; }
    curl_close($ch);
    $data = json_decode($response, true);
    if (is_array($data)) {
        if (!empty($data['SessionToken'])) return $data['SessionToken'];
        if (!empty($data['data']['SessionToken'])) return $data['data']['SessionToken'];
        if (!empty($data['sessionToken'])) return $data['sessionToken'];
    }
    return false;
}

/**
 * GET de transacciones con paginación.
 */
function nb_getAllTransactions($apiBaseUrl, $aptoken, $accountId, $fromDate = null, $toDate = null) {
    $page = 1;
    $pageSize = 50;
    $allData = [];
    do {
        $params = [
            'page'      => $page,
            'page_size' => $pageSize
        ];
        if ($accountId) $params['accountId'] = $accountId;
        if ($fromDate)  $params['from_date'] = $fromDate;
        if ($toDate)    $params['to_date']   = $toDate;

        $url = rtrim($apiBaseUrl, '/') . '/api/nb/wallet/transactions?' . http_build_query($params);
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER     => [
                'Content-Type: application/json',
                'Accept: application/json',
                'aptoken: ' . $aptoken
            ]
        ]);
        $response = curl_exec($ch);
        if (curl_errno($ch)) { curl_close($ch); break; }
        curl_close($ch);
        $result = json_decode($response, true);
        $data = $result['data'] ?? [];
        $allData = array_merge($allData, $data);
        $page++;
        $hasMore = count($data) >= $pageSize;
    } while ($hasMore);
    return $allData;
}

/**
 * Normaliza el nombre de red esperado por la API (USDT_BSC, USDT_ETH, etc.).
 */
function nb_normalizeNetwork($currency, $network) {
    $c = strtoupper(trim($currency));
    $n = strtoupper(trim($network));
    $n = str_replace('-', '_', $n);

    // Sinónimos comunes
    $map = [
        'BEP20' => 'BSC',
        'TRC20' => 'TRC',
        'ERC20' => 'ETH',
    ];
    if (isset($map[$n])) $n = $map[$n];

    if (preg_match('/^' . preg_quote($c, '/') . '_(BSC|ETH|TRC|SOL)$/', $n)) {
        return $n;
    }
    if (in_array($n, ['BSC','ETH','TRC','SOL'], true)) {
        return $c . '_' . $n;
    }
    if (preg_match('/^([A-Z]+)(BSC|ETH|TRC|SOL)$/', $n, $m)) {
        return $m[1] . '_' . $m[2];
    }
    return $n;
}

/**
 * GET: direcciones whitelisteadas de una cuenta.
 */
function nb_getWhitelistedAddresses($apiBaseUrl, $aptoken, $accountId, $search = null) {
    if (!$accountId) return [];
    $params = ['account_id' => (int)$accountId];
    if (!empty($search)) $params['search'] = $search;

    $url = rtrim($apiBaseUrl, '/') . '/api/nb/wallet/crypto/whitelist-addresses?' . http_build_query($params);
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'Accept: application/json',
            'aptoken: ' . $aptoken
        ]
    ]);
    $response = curl_exec($ch);
    if (curl_errno($ch)) { curl_close($ch); return false; }
    curl_close($ch);
    $result = json_decode($response, true);
    if (is_array($result) && isset($result['status']) && strtolower($result['status']) === 'success') {
        return $result['data'] ?? [];
    }
    return false;
}

/**
 * Valida whitelist: dirección + verificada + moneda + red.
 * Retorna ['ok'=>bool, 'entry'=>array|null, 'why'=>string]
 */
function nb_isAddressWhitelisted(array $whitelist, $address, $network, $currency) {
    $address  = strtolower(trim($address));
    $networkN = nb_normalizeNetwork($currency, $network);
    $currencyU= strtoupper(trim($currency));

    foreach ($whitelist as $entry) {
        $entryAddr = isset($entry['address']) ? strtolower(trim($entry['address'])) : '';
        $entryNet  = isset($entry['network']) ? strtoupper(trim($entry['network'])) : '';
        $entryCur  = isset($entry['currency'])? strtoupper(trim($entry['currency'])): '';
        $verified  = !empty($entry['verified']);

        if ($entryAddr === $address) {
            if (!$verified) {
                return ['ok'=>false, 'entry'=>$entry, 'why'=>'La dirección existe en la whitelist, pero aún no está verificada.'];
            }
            if ($entryCur !== $currencyU) {
                return ['ok'=>false, 'entry'=>$entry, 'why'=>"La dirección está whitelisteada para {$entryCur}, no para {$currencyU}."];
            }
            if ($entryNet !== $networkN) {
                return ['ok'=>false, 'entry'=>$entry, 'why'=>"La dirección está whitelisteada para la red {$entryNet}, no para {$networkN}."];
            }
            return ['ok'=>true, 'entry'=>$entry, 'why'=>'OK'];
        }
    }
    return ['ok'=>false, 'entry'=>null, 'why'=>'La dirección no está en la whitelist.'];
}

/**
 * POST: agrega dirección a whitelist.
 */
function nb_addWhitelistedAddress($apiBaseUrl, $aptoken, $accountId, $currency, $network, $address, $label, $memo = null, $otp) {
    $payload = [
        'account_id' => (int)$accountId,
        'currency'   => strtoupper(trim($currency)),
        'network'    => nb_normalizeNetwork($currency, $network),
        'address'    => trim($address),
        'label'      => $label,
        'otp'        => $otp
    ];
    if (!empty($memo)) $payload['memo_or_tag'] = $memo;

    $url = rtrim($apiBaseUrl, '/') . '/api/nb/wallet/crypto/whitelist-addresses';
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/json',
            'Accept: application/json',
            'aptoken: ' . $aptoken
        ],
        CURLOPT_POSTFIELDS     => json_encode($payload)
    ]);
    $response = curl_exec($ch);
    if (curl_errno($ch)) { curl_close($ch); return false; }
    curl_close($ch);
    return json_decode($response, true);
}

/**
 * POST: confirma dirección whitelist por SMS code.
 */
function nb_confirmWhitelistedAddress($apiBaseUrl, $aptoken, $whitelistedAddressId, $accountId, $smsCode) {
    $url = rtrim($apiBaseUrl, '/') . '/api/nb/wallet/crypto/whitelist-addresses/' . $whitelistedAddressId . '/verification';
    $payload = [
        'account_id' => (int)$accountId,
        'sms_code'   => $smsCode
    ];
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/json',
            'Accept: application/json',
            'aptoken: ' . $aptoken
        ],
        CURLOPT_POSTFIELDS     => json_encode($payload)
    ]);
    $response = curl_exec($ch);
    if (curl_errno($ch)) { curl_close($ch); return false; }
    curl_close($ch);
    return json_decode($response, true);
}

/** Utilidad: parsea headers crudos */
function nb_parseHeaders($raw) {
    $headers = [];
    $lines = preg_split("/\r\n|\n|\r/", $raw);
    foreach ($lines as $line) {
        if (strpos($line, ':') !== false) {
            [$k,$v] = explode(':', $line, 2);
            $headers[trim(strtolower($k))] = trim($v);
        }
    }
    return $headers;
}

/** POST JSON con captura de http_code, headers, body y JSON */
function nb_http_post_json($url, array $headers, array $payload) {
    $ch = curl_init($url);
    $body = json_encode($payload);
    $headers[] = 'Accept: application/json';
    $headers[] = 'Expect:'; // evita 100-continue

    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_HTTPHEADER     => $headers,
        CURLOPT_POSTFIELDS     => $body,
        CURLOPT_HEADER         => true,
        CURLOPT_FAILONERROR    => false,
        CURLOPT_TIMEOUT        => 30,
    ]);

    $resp = curl_exec($ch);
    $curlErr = curl_error($ch);
    $info = curl_getinfo($ch);
    curl_close($ch);

    $headerSize = $info['header_size'] ?? 0;
    $rawHeaders = substr($resp, 0, $headerSize);
    $rawBody    = substr($resp, $headerSize);

    $json = json_decode($rawBody, true);
    return [
        'http_code' => $info['http_code'] ?? 0,
        'headers'   => nb_parseHeaders($rawHeaders),
        'body_raw'  => $rawBody,
        'json'      => is_array($json) ? $json : null,
        'curl_err'  => $curlErr,
        'req_body'  => $body,
        'url'       => $url,
    ];
}

/**
 * POST: CreateCryptoWithdraw con diagnóstico.
 * - amount se envía como string decimal (hasta 8 decimales).
 * - network se normaliza.
 * Retorna paquete con http_code/headers/body/json.
 */
function nb_createCryptoWithdraw($apiBaseUrl, $aptoken, $accountId, $currency, $network, $address, $amount, $memoTag = null, $otp = null) {
    $url = rtrim($apiBaseUrl, '/') . '/api/nb/wallet/crypto/withdrawal';

    // Evita problemas de float binario
    $amountStr = rtrim(rtrim(number_format((float)$amount, 8, '.', ''), '0'), '.');
    if ($amountStr === '') $amountStr = '0';

    $payload = [
        'account_id' => (int)$accountId,
        'currency'   => strtoupper(trim($currency)),
        'network'    => nb_normalizeNetwork($currency, $network),
        'address'    => trim($address),
        'amount'     => $amountStr,
    ];
    if (!empty($memoTag)) $payload['memo_or_tag'] = $memoTag;
    if (!empty($otp))     $payload['otp']         = $otp;

    $res = nb_http_post_json($url, [
        'Content-Type: application/json',
        'aptoken: ' . $aptoken
    ], $payload);

    return $res;
}

/**
 * Interpreta respuesta de retiro (paquete nb_http_post_json) y construye mensaje.
 * Devuelve [mensaje_html, reintentar_bool]
 */
function nb_handleWithdrawResponse($res) {
    if (!is_array($res)) {
        return ['Error al realizar el retiro: respuesta inesperada del servidor.', false];
    }

    $http = $res['http_code'] ?? 0;
    $json = $res['json'];
    $body = $res['body_raw'] ?? '';
    $hdrs = $res['headers'] ?? [];
    $rid  = $hdrs['x-request-id'] ?? $hdrs['x-correlation-id'] ?? null;

    $status = is_array($json) && isset($json['status']) ? strtolower((string)$json['status']) : null;
    $msg    = is_array($json) ? ($json['message'] ?? $json['detail'] ?? $json['error'] ?? '') : '';

    $fieldErrorsHtml = '';
    if (is_array($json) && !empty($json['errors']) && is_array($json['errors'])) {
        $fieldErrorsHtml .= '<ul>';
        foreach ($json['errors'] as $k => $v) {
            if (is_array($v)) $v = implode('; ', $v);
            $fieldErrorsHtml .= '<li><strong>' . htmlspecialchars((string)$k) . ':</strong> ' . htmlspecialchars((string)$v) . '</li>';
        }
        $fieldErrorsHtml .= '</ul>';
    }

    // Éxito
    if ($http >= 200 && $http < 300 && $status === 'success') {
        $id = '';
        if (isset($json['data'])) {
            $id = is_array($json['data']) && isset($json['data']['id']) ? $json['data']['id']
                : (is_string($json['data']) ? $json['data'] : '');
        }
        $ok = 'Retiro solicitado con éxito.' . ($id ? ' ID: ' . htmlspecialchars($id) : '');
        return [$ok, false];
    }

    // Causas comunes
    $reauth = false;
    $extra  = '';
    if ($http === 401 || stripos($msg, 'unauthorized') !== false || stripos($msg, 'token') !== false) {
        $reauth = true;
        $extra = ' (sesión expirada o token inválido; se reintentará)';
    } elseif ($http === 403) {
        $extra = ' (permiso insuficiente; revisa NotbankWithdraw/KYC/limitaciones de cuenta)';
    } elseif ($http === 422) {
        $extra = ' (error de validación; revisa campos abajo)';
    } elseif ($http === 409) {
        $extra = ' (posible idempotencia o solicitud duplicada)';
    } elseif ($http === 400) {
        $extra = ' (parámetros inválidos)';
    } elseif ($http >= 500) {
        $extra = ' (error interno del proveedor; suele ocurrir por combinación inválida red/moneda/dirección)';
    }

    // Bloque de diagnóstico expandible
    $diag  = '<details style="margin-top:8px;"><summary>Diagnóstico</summary><pre style="white-space:pre-wrap;">';
    $diag .= "HTTP: {$http}\n";
    if (!empty($msg)) $diag .= "message/detail: {$msg}\n";
    if (!empty($fieldErrorsHtml)) $diag .= "</pre>{$fieldErrorsHtml}<pre style=\"white-space:pre-wrap;\">";
    if (!empty($rid)) $diag .= "request-id: {$rid}\n";
    $diag .= "response:\n" . $body . "\n";
    $diag .= "</pre></details>";

    $human = 'Error al realizar el retiro' . $extra . '.';
    if (!empty($msg)) $human .= ' ' . htmlspecialchars($msg);

    return [$human . $diag, $reauth];
}

// ------ Lógica del script ------

$fromDate     = isset($_POST['fromDate']) && trim($_POST['fromDate']) !== '' ? trim($_POST['fromDate']) : null;
$toDate       = isset($_POST['toDate'])   && trim($_POST['toDate'])   !== '' ? trim($_POST['toDate'])   : null;
$transactions = [];
$message      = '';

$action = $_POST['action'] ?? null;

if ($action) {
    switch ($action) {
        case 'withdraw':
            $withdrawAccountId = (int)($_POST['source_account_id'] ?? 0);
            $currency   = $_POST['currency'] ?? '';
            $amount     = (float) ($_POST['amount'] ?? 0);
            $network    = $_POST['network'] ?? '';
            $address    = $_POST['address'] ?? '';
            $memoTag    = $_POST['memo'] ?? null;
            $withdrawOtp= $_POST['otp'] ?? null;

            if ($withdrawAccountId && $currency && $amount > 0 && $network && $address) {
                $aptoken = $_SESSION['aptoken'] ?? nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId);
                if ($aptoken) $_SESSION['aptoken'] = $aptoken;

                if ($aptoken) {
                    // 1) Normaliza red
                    $networkN = nb_normalizeNetwork($currency, $network);

                    // 2) Valida whitelist estricta
                    $whitelist = nb_getWhitelistedAddresses($apiBaseUrl, $aptoken, $withdrawAccountId);
                    if ($whitelist === false) {
                        $message = 'No se pudo obtener la lista de direcciones whitelisteadas. Revisa permisos.';
                        break;
                    }
                    $wlCheck = nb_isAddressWhitelisted($whitelist, $address, $networkN, $currency);
                    if (!$wlCheck['ok']) {
                        $message = 'No se puede retirar: ' . htmlspecialchars($wlCheck['why']);
                        break;
                    }

                    // 3) Ejecuta retiro con diagnóstico
                    $res = nb_createCryptoWithdraw($apiBaseUrl, $aptoken, $withdrawAccountId, $currency, $networkN, $address, $amount, $memoTag, $withdrawOtp);
                    list($msg, $reauth) = nb_handleWithdrawResponse($res);

                    if ($reauth) {
                        unset($_SESSION['aptoken']);
                        $aptoken = nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId);
                        if ($aptoken) {
                            $_SESSION['aptoken'] = $aptoken;
                            $resRetry = nb_createCryptoWithdraw($apiBaseUrl, $aptoken, $withdrawAccountId, $currency, $networkN, $address, $amount, $memoTag, $withdrawOtp);
                            list($msg2, $reauth2) = nb_handleWithdrawResponse($resRetry);
                            $message = $msg2;
                        } else {
                            $message = 'No se pudo reautenticar. Vuelve a intentarlo.';
                        }
                    } else {
                        $message = $msg;
                    }
                } else {
                    $message = 'Error de autenticación al obtener el token para el retiro.';
                }
            } else {
                $message = 'Completa todos los campos obligatorios para el retiro.';
            }
            break;

        case 'add_whitelist':
            $wlAccountId = (int)($_POST['wl_account_id'] ?? 0);
            $wlCurrency  = $_POST['wl_currency'] ?? '';
            $wlNetwork   = $_POST['wl_network'] ?? '';
            $wlAddress   = $_POST['wl_address'] ?? '';
            $wlLabel     = $_POST['wl_label'] ?? '';
            $wlMemo      = $_POST['wl_memo'] ?? null;
            $wlOtp       = $_POST['wl_otp'] ?? '';

            if ($wlAccountId && $wlCurrency && $wlNetwork && $wlAddress && $wlLabel && $wlOtp) {
                $aptokenWL = $_SESSION['aptoken'] ?? nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId);
                if ($aptokenWL) $_SESSION['aptoken'] = $aptokenWL;

                if ($aptokenWL) {
                    $respAdd = nb_addWhitelistedAddress($apiBaseUrl, $aptokenWL, $wlAccountId, $wlCurrency, $wlNetwork, $wlAddress, $wlLabel, $wlMemo, $wlOtp);
                    if (is_array($respAdd) && isset($respAdd['status']) && strtolower($respAdd['status']) === 'success') {
                        $newId = $respAdd['data']['id'] ?? '';
                        $message = 'Dirección whitelisteada creada con éxito. ID: ' . htmlspecialchars($newId) . '. Ahora debes confirmarla con el código SMS usando el formulario de confirmación.';
                    } else {
                        $errMsgWL = $respAdd['message'] ?? ($respAdd['detail'] ?? ($respAdd['error'] ?? 'Error desconocido'));
                        $message = 'Error al crear la dirección whitelisteada: ' . htmlspecialchars($errMsgWL);
                    }
                } else {
                    $message = 'No se pudo autenticar para agregar la dirección.';
                }
            } else {
                $message = 'Completa todos los campos obligatorios para agregar una dirección a la whitelist.';
            }
            break;

        case 'confirm_whitelist':
            $confirmId        = $_POST['confirm_id'] ?? '';
            $confirmCode      = $_POST['confirm_code'] ?? '';
            $confirmAccountId = (int)($_POST['confirm_account_id'] ?? 0);

            if ($confirmId && $confirmCode && $confirmAccountId) {
                $aptokenConfirm = $_SESSION['aptoken'] ?? nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId);
                if ($aptokenConfirm) $_SESSION['aptoken'] = $aptokenConfirm;

                if ($aptokenConfirm) {
                    $respConfirm = nb_confirmWhitelistedAddress($apiBaseUrl, $aptokenConfirm, $confirmId, $confirmAccountId, $confirmCode);
                    if (is_array($respConfirm) && isset($respConfirm['status']) && strtolower($respConfirm['status']) === 'success') {
                        $message = 'Dirección confirmada con éxito. Ya puedes utilizarla para retirar fondos.';
                    } else {
                        $errMsgC = $respConfirm['message'] ?? ($respConfirm['detail'] ?? ($respConfirm['error'] ?? 'Error desconocido'));
                        $message = 'Error al confirmar la dirección: ' . htmlspecialchars($errMsgC);
                    }
                } else {
                    $message = 'No se pudo autenticar para confirmar la dirección.';
                }
            } else {
                $message = 'Proporciona el ID de la dirección y el código SMS para confirmarla.';
            }
            break;

        case 'check_whitelist':
            $checkAccountId = (int)($_POST['check_account_id'] ?? 0);
            $checkCurrency  = $_POST['check_currency'] ?? '';
            $checkNetwork   = $_POST['check_network'] ?? '';
            $checkAddress   = $_POST['check_address'] ?? '';

            if ($checkAccountId && $checkAddress) {
                $aptokenCheck = $_SESSION['aptoken'] ?? nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId);
                if ($aptokenCheck) $_SESSION['aptoken'] = $aptokenCheck;

                if ($aptokenCheck) {
                    $wlList = nb_getWhitelistedAddresses($apiBaseUrl, $aptokenCheck, $checkAccountId);
                    if ($wlList === false) {
                        $message = 'No se pudo obtener la lista de direcciones whitelisteadas. Asegúrate de que tu API Key tenga los permisos necesarios.';
                    } else {
                        $netN = $checkNetwork ? nb_normalizeNetwork($checkCurrency ?: 'USDT', $checkNetwork) : $checkNetwork;
                        $wl = nb_isAddressWhitelisted($wlList, $checkAddress, $netN, $checkCurrency ?: 'USDT');
                        $addrDisplay = htmlspecialchars($checkAddress);
                        if ($wl['ok']) {
                            $message = 'La dirección ' . $addrDisplay . ' está en la whitelist de la cuenta y ha sido verificada.';
                        } else {
                            $message  = 'La dirección ' . $addrDisplay . ' NO está en la whitelist o no ha sido confirmada. ';
                            $message .= htmlspecialchars($wl['why']);
                            $message .= ' Recuerda: las direcciones de depósito de Notbank no cuentan como whitelist de retiros.';
                        }
                    }
                } else {
                    $message = 'No se pudo autenticar para comprobar la whitelist.';
                }
            } else {
                $message = 'Proporciona al menos la cuenta y la dirección para verificar en la whitelist.';
            }
            break;

        default:
            // no-op
            break;
    }
}

// Listado de transacciones (últimas 5)
$aptokenForList = $_SESSION['aptoken'] ?? nb_authenticate($apiBaseUrl, $apiKey, $apiSecret, $userId);
if ($aptokenForList) $_SESSION['aptoken'] = $aptokenForList;

if ($aptokenForList) {
    $transactions = nb_getAllTransactions($apiBaseUrl, $aptokenForList, $accountId, $fromDate ?: null, $toDate ?: null);
    usort($transactions, function ($a, $b) {
        $dateA = isset($a['created_at']) ? strtotime($a['created_at']) : 0;
        $dateB = isset($b['created_at']) ? strtotime($b['created_at']) : 0;
        return $dateB <=> $dateA;
    });
    $transactions = array_slice($transactions, 0, 5);
} else {
    $message = $message ?: 'No se pudo autenticar para obtener las transacciones.';
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Historial y retiros de criptomonedas - Notbank</title>
    <style>
        body { font-family: "Segoe UI", Tahoma, sans-serif; margin:0; padding:0; background:#f5f7fa; color:#333; }
        .container { max-width:1000px; margin:0 auto; padding:20px; }
        h1 { margin-top:0; color:#2a3f54; }
        p.description { color:#556871; margin-bottom:20px; }
        .card { background:#fff; border-radius:8px; box-shadow:0 2px 6px rgba(0,0,0,.05); padding:20px; margin-bottom:20px; }
        form { margin-bottom:0; }
        .form-row { display:flex; flex-wrap:wrap; gap:10px; }
        .form-group { flex:1; min-width:180px; display:flex; flex-direction:column; }
        label { font-size:.85rem; margin-bottom:4px; color:#3a3a3a; }
        input[type="text"], input[type="number"], input[type="password"], select {
            padding:8px 10px; border:1px solid #ccc; border-radius:4px; font-size:.9rem;
        }
        button { padding:8px 16px; background:#5b9bd5; border:none; border-radius:4px; color:#fff; font-weight:bold; cursor:pointer; transition:.2s; }
        button:hover { background:#4a86c5; }
        .message { margin-top:20px; padding:12px 16px; border-radius:4px; }
        .message.success { background:#e0f5e9; color:#0a572c; }
        .message.error { background:#fbeaea; color:#a94442; }
        table { width:100%; border-collapse:collapse; margin-top:10px; }
        th, td { padding:10px; border-bottom:1px solid #eee; font-size:.9rem; }
        th { background:#f0f3f6; color:#445364; text-align:left; font-weight:600; }
        tr:nth-child(even) td { background:#fafcfe; }
        tr:hover td { background:#eef4fc; }
        .withdraw-form { background:#f6f8fa; border:1px solid #e1e7ef; border-radius:4px; padding:8px; }
        .withdraw-form label { margin-bottom:2px; font-size:.8rem; color:#445364; }
        .withdraw-form input[type="text"], .withdraw-form input[type="number"] {
            padding:6px; border:1px solid #ccd5e0; border-radius:4px; font-size:.8rem;
        }
        .withdraw-form button { margin-top:6px; width:100%; font-size:.9rem; }
        .transaction-list { display:flex; flex-direction:column; align-items:center; gap:16px; }
        .tx-item { display:flex; flex-wrap:wrap; gap:16px; border:1px solid #e1e7ef; border-radius:8px; padding:16px; background:#fafcfe; width:100%; max-width:700px; }
        .tx-info { flex:1 1 250px; min-width:200px; }
        .tx-form { flex:1 1 250px; min-width:200px; }
        .tx-info p { margin:4px 0; font-size:.85rem; color:#445364; }
        details summary { cursor:pointer; font-weight:600; }
        details pre { background:#fff; border:1px solid #e1e7ef; padding:10px; border-radius:6px; overflow:auto; }
    </style>
</head>
<body>
<div class="container">
    <h1>Historial y retiros de criptomonedas</h1>
    <p class="description">
        Consulta los movimientos y solicita retiros. Las direcciones destino deben estar en la whitelist con la <strong>misma moneda y red</strong>.
        Si tu cuenta requiere 2FA para retiros, ingresa el código en el campo <strong>OTP</strong>.
    </p>

    <div class="card">
        <h2>Filtrar historial</h2>
        <form method="post">
            <div class="form-row">
                <div class="form-group">
                    <label for="fromDate">Fecha desde (dd/mm/yyyy)</label>
                    <input type="text" id="fromDate" name="fromDate" value="<?php echo htmlspecialchars($fromDate ?? ''); ?>" placeholder="Opcional">
                </div>
                <div class="form-group">
                    <label for="toDate">Fecha hasta (dd/mm/yyyy)</label>
                    <input type="text" id="toDate" name="toDate" value="<?php echo htmlspecialchars($toDate ?? ''); ?>" placeholder="Opcional">
                </div>
                <div class="form-group" style="align-self:flex-end;">
                    <button type="submit">Buscar</button>
                </div>
            </div>
        </form>
    </div>

    <?php if (!empty($message)): ?>
        <div class="message <?php echo (strpos($message, 'éxito') !== false ? 'success' : 'error'); ?>">
            <?php echo $message; ?>
        </div>
    <?php endif; ?>

    <div class="card">
        <h2>Transacciones encontradas</h2>
        <?php if (!empty($transactions)): ?>
            <div class="transaction-list">
                <?php foreach ($transactions as $tx): ?>
                    <?php
                    $txDate    = htmlspecialchars($tx['created_at'] ?? '');
                    $txCurrencyRaw = $tx['currency'] ?? '';
                    $txAddress = htmlspecialchars($tx['address'] ?? '');
                    $txAmount  = $tx['amount'] ?? '';
                    $txType    = (isset($tx['direction']) && (int)$tx['direction'] === 1) ? 'Entrada' : 'Salida';
                    $txHash    = htmlspecialchars($tx['hash'] ?? '');

                    // Deducir red por patrón (USDTBSC -> USDT_BSC)
                    $defaultNetwork = '';
                    $txCurrency = $txCurrencyRaw;
                    if (!empty($txCurrencyRaw) && preg_match('/([A-Za-z]+)(BSC|ETH|TRC|SOL)/', $txCurrencyRaw, $m)) {
                        $defaultNetwork = $m[1] . '_' . $m[2];
                        $txCurrency = $m[1];
                    }
                    ?>
                    <div class="tx-item">
                        <div class="tx-info">
                            <p><strong>Fecha:</strong> <?php echo $txDate; ?></p>
                            <p><strong>Moneda:</strong> <?php echo htmlspecialchars($txCurrencyRaw); ?></p>
                            <p><strong>Dirección:</strong> <?php echo $txAddress; ?></p>
                            <p><strong>Monto:</strong> <?php echo htmlspecialchars($txAmount); ?></p>
                            <p><strong>Tipo:</strong> <?php echo $txType; ?></p>
                            <p><strong>Hash:</strong> <?php echo $txHash; ?></p>
                        </div>
                        <div class="tx-form">
                            <form method="post" class="withdraw-form">
                                <input type="hidden" name="action" value="withdraw">
                                <input type="hidden" name="source_account_id" value="<?php echo (int)($tx['account_id'] ?? $accountId); ?>">
                                <input type="hidden" name="currency" value="<?php echo htmlspecialchars($txCurrency); ?>">
                                <div class="form-group">
                                    <label>Monto a retirar</label>
                                    <input type="number" step="0.00000001" name="amount" value="<?php echo htmlspecialchars($txAmount); ?>" required>
                                </div>
                                <div class="form-group">
                                    <label>Red</label>
                                    <input type="text" name="network" value="<?php echo htmlspecialchars($defaultNetwork); ?>" placeholder="USDT_BSC / USDT_ETH / USDT_TRC" required>
                                </div>
                                <div class="form-group">
                                    <label>Dirección destino</label>
                                    <input type="text" name="address" placeholder="0x... / Dirección de la red" required>
                                </div>
                                <div class="form-group">
                                    <label>Memo/Tag (opcional)</label>
                                    <input type="text" name="memo" placeholder="">
                                </div>
                                <div class="form-group">
                                    <label>OTP (opcional)</label>
                                    <input type="text" name="otp" placeholder="Código 2FA si tu cuenta lo exige">
                                </div>
                                <button type="submit">Enviar</button>
                            </form>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <p>No se encontraron transacciones para los filtros seleccionados.</p>
        <?php endif; ?>
    </div>

    <div class="card">
        <h2>Administrar whitelist</h2>
        <p class="description">Agrega/Confirma direcciones de retiro. La moneda y la red deben coincidir exactamente con lo que usarás en el retiro.</p>

        <h3>Agregar dirección</h3>
        <form method="post" class="withdraw-form" style="max-width:600px;">
            <input type="hidden" name="action" value="add_whitelist">
            <div class="form-row">
                <div class="form-group">
                    <label>Cuenta ID</label>
                    <input type="number" name="wl_account_id" value="<?php echo (int)$accountId; ?>" required>
                </div>
                <div class="form-group">
                    <label>Moneda</label>
                    <input type="text" name="wl_currency" placeholder="USDT, BTC" required>
                </div>
                <div class="form-group">
                    <label>Red</label>
                    <input type="text" name="wl_network" placeholder="USDT_BSC, BTC_MAINNET" required>
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label>Dirección</label>
                    <input type="text" name="wl_address" placeholder="0x... / Dirección" required>
                </div>
                <div class="form-group">
                    <label>Etiqueta</label>
                    <input type="text" name="wl_label" placeholder="Nombre descriptivo" required>
                </div>
                <div class="form-group">
                    <label>Memo/Tag (opcional)</label>
                    <input type="text" name="wl_memo" placeholder="">
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label>OTP / 2FA</label>
                    <input type="text" name="wl_otp" placeholder="Código 2FA" required>
                </div>
                <div class="form-group" style="align-self:flex-end;">
                    <button type="submit">Crear dirección</button>
                </div>
            </div>
        </form>

        <h3>Confirmar dirección</h3>
        <form method="post" class="withdraw-form" style="max-width:600px;">
            <input type="hidden" name="action" value="confirm_whitelist">
            <div class="form-row">
                <div class="form-group">
                    <label>ID de dirección whitelisteada</label>
                    <input type="text" name="confirm_id" placeholder="ID devuelto al crear" required>
                </div>
                <div class="form-group">
                    <label>Cuenta ID</label>
                    <input type="number" name="confirm_account_id" value="<?php echo (int)$accountId; ?>" required>
                </div>
                <div class="form-group">
                    <label>Código SMS</label>
                    <input type="text" name="confirm_code" placeholder="Código SMS" required>
                </div>
            </div>
            <div class="form-row">
                <div class="form-group" style="align-self:flex-end;">
                    <button type="submit">Confirmar dirección</button>
                </div>
            </div>
        </form>

        <h3>Consultar dirección whitelisteada</h3>
        <form method="post" class="withdraw-form" style="max-width:600px;">
            <input type="hidden" name="action" value="check_whitelist">
            <div class="form-row">
                <div class="form-group">
                    <label>Cuenta ID</label>
                    <input type="number" name="check_account_id" value="<?php echo (int)$accountId; ?>" required>
                </div>
                <div class="form-group">
                    <label>Moneda (opcional)</label>
                    <input type="text" name="check_currency" placeholder="USDT, BTC">
                </div>
                <div class="form-group">
                    <label>Red (opcional)</label>
                    <input type="text" name="check_network" placeholder="USDT_BSC, BTC_MAINNET">
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label>Dirección</label>
                    <input type="text" name="check_address" placeholder="0x... / Dirección" required>
                </div>
                <div class="form-group" style="align-self:flex-end;">
                    <button type="submit">Verificar</button>
                </div>
            </div>
        </form>
    </div>
</div>
</body>
</html>
