

<?php
/*
 * Notbank transaction history viewer
 *
 * This script authenticates to the Notbank API using a public/private key
 * pair and user ID, obtains a session token (aptoken), and then fetches
 * historical transactions for the account.  Users can filter by date
 * range using DD/MM/YYYY format.  All other credentials are fixed in
 * this file for simplicity.
 *
 * IMPORTANT: You must confirm that the base URL ($apiBaseUrl) matches
 * the environment provided by your Notbank deployment.  The
 * authentication request requires a signature computed from
 * Nonce + UserId + ApiKey using the secret key.
 */

// -----------------------------------------------------------------------------
// Configuration (replace with your actual credentials)
// -----------------------------------------------------------------------------
$apiBaseUrl = 'https://api.notbank.exchange';
$apiKey     = '40b266e4f2f010631ddfb26cda4d915a'; // Llave pública (APIKey)
$secretKey  = '5bc92a8df76d2d88248590e0fef53ecc'; // Llave privada (secret)
$userId     = '1081';                             // User ID
$accountId  = 4231; // <-- Poner tu ID de cuenta real
/**
 * Authenticate with Notbank and return a session token.
 *
 * The signature is generated by concatenating nonce, userId and apiKey,
 * then hashing with HMAC-SHA256 using the secret key.
 *
 * @param string $baseUrl
 * @param string $apiKey
 * @param string $secretKey
 * @param string $userId
 * @return string|false The session token (aptoken) on success, false on failure.
 */
function authenticateAndGetToken($baseUrl, $apiKey, $secretKey, $userId) {
    // Use a nonce based on current time in milliseconds
    $nonce = (string) round(microtime(true) * 1000);
    $message = $nonce . $userId . $apiKey;
    $signature = hash_hmac('sha256', $message, $secretKey);

    // Build URL and headers for the GET request
    $url = rtrim($baseUrl, '/') . '/AP/AuthenticateUser';
    $headers = [
        'APIKey: ' . $apiKey,
        'Signature: ' . $signature,
        'UserId: ' . $userId,
        'Nonce: ' . $nonce,
        'Content-Type: application/json'
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    if ($response === false) {
        curl_close($ch);
        return false;
    }
    curl_close($ch);
    $data = json_decode($response, true);
    // The session token is returned in SessionToken
    if (isset($data['SessionToken'])) {
        return $data['SessionToken'];
    }
    return false;
}

/**
 * Retrieve all transactions using the session token.
 *
 * @param string $baseUrl
 * @param string $sessionToken
 * @param string|null $fromDate
 * @param string|null $toDate
 * @return array
 */
function fetchTransactions($baseUrl, $sessionToken, $fromDate = null, $toDate = null) {
    $allTransactions = [];
    $page = 1;
    do {
        // Assemble query params; only include date filters if provided
        $params = [
            'page' => $page,
            'page_size' => 50
        ];
        if (!empty($fromDate)) {
            $params['from_date'] = $fromDate;
        }
        if (!empty($toDate)) {
            $params['to_date'] = $toDate;
        }
        $url = rtrim($baseUrl, '/') . '/api/nb/wallet/transactions?' . http_build_query($params);

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'aptoken: ' . $sessionToken,
            'Content-Type: application/json'
        ]);
        $response = curl_exec($ch);
        if ($response === false) {
            curl_close($ch);
            break;
        }
        curl_close($ch);
        $result = json_decode($response, true);
        if (!is_array($result) || !isset($result['status']) || strtolower($result['status']) !== 'success') {
            break;
        }
        $data = isset($result['data']) && is_array($result['data']) ? $result['data'] : [];
        $allTransactions = array_merge($allTransactions, $data);
        // Continue if the page returned data, otherwise stop
        $hasMore = count($data) > 0;
        $page++;
    } while ($hasMore);
    return $allTransactions;
}

/**
 * Helper to escape and format transaction fields for table output.
 */
function h($value) {
    return htmlspecialchars((string) $value, ENT_QUOTES, 'UTF-8');
}

// If the form is submitted, process the request
$transactions = [];
$authError = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $fromDateInput = isset($_POST['fromDate']) && trim($_POST['fromDate']) !== '' ? trim($_POST['fromDate']) : null;
    $toDateInput   = isset($_POST['toDate'])   && trim($_POST['toDate'])   !== '' ? trim($_POST['toDate'])   : null;

    // Step 1: authenticate and get session token
    $token = authenticateAndGetToken($apiBaseUrl, $apiKey, $secretKey, $userId);
    if ($token === false) {
        $authError = 'Error de autenticación. Verifica credenciales, firma y endpoint.';
    } else {
        // Step 2: fetch transactions with optional date filters
        $transactions = fetchTransactions($apiBaseUrl, $token, $fromDateInput, $toDateInput);
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Historial de Transacciones Notbank</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        table { border-collapse: collapse; width: 100%; margin-top: 20px; }
        th, td { border: 1px solid #ccc; padding: 8px; text-align: left; }
        th { background-color: #f4f4f4; }
        form { max-width: 400px; margin-bottom: 20px; }
        label { display: block; margin: 8px 0 4px; }
        input, button { width: 100%; padding: 8px; margin-bottom: 10px; }
        .error { color: red; }
    </style>
</head>
<body>
<h1>Historial de Transacciones Notbank</h1>
<p>Rellena las fechas opcionales para filtrar. Deja en blanco para ver todas las transacciones.</p>
<form method="post">
    <label for="fromDate">Fecha desde (dd/mm/yyyy):</label>
    <input type="text" id="fromDate" name="fromDate" placeholder="opcional">
    <label for="toDate">Fecha hasta (dd/mm/yyyy):</label>
    <input type="text" id="toDate" name="toDate" placeholder="opcional">
    <button type="submit">Consultar</button>
</form>
<?php
if (!empty($authError)) {
    echo '<p class="error">' . h($authError) . '</p>';
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (empty($transactions)) {
        echo '<p>No se encontraron transacciones para el criterio proporcionado.</p>';
    } else {
        echo '<h2>Total transacciones: ' . count($transactions) . '</h2>';
        echo '<table>';
        echo '<tr><th>Fecha</th><th>Moneda</th><th>Dirección</th><th>Monto</th><th>Tipo</th><th>Hash</th></tr>';
        foreach ($transactions as $tx) {
            $direction = isset($tx['direction']) && $tx['direction'] == 1 ? 'Entrada' : (isset($tx['direction']) && $tx['direction'] == 2 ? 'Salida' : 'Otro');
            echo '<tr>';
            echo '<td>' . h($tx['created_at'] ?? '') . '</td>';
            echo '<td>' . h($tx['currency'] ?? '') . '</td>';
            echo '<td>' . h($tx['address'] ?? '') . '</td>';
            echo '<td>' . h($tx['amount'] ?? '') . '</td>';
            echo '<td>' . h($direction) . '</td>';
            echo '<td>' . h($tx['hash'] ?? '') . '</td>';
            echo '</tr>';
        }
        echo '</table>';
    }
}
?>
</body>
</html>